"""Class to run test scripts on dialogs."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import importlib.util
import os
import re

# 2. Third party modules
from PySide2.QtCore import QModelIndex, Qt
from PySide2.QtGui import QAccessible
from PySide2.QtTest import QTest
from PySide2.QtWidgets import QApplication, QDialog

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.testing.script_saver_dialog import ScriptSaverDialog


class GuiTester:
    """Class to run test scripts on dialogs."""
    def __init__(self, qapp):
        """Constructor.

        Args:
            qapp (QApplication): The Qt app
        """
        self.qapp = qapp  # The QApplication
        self.xpath = None  # The string xpath to the widget from the script
        self.curr_object = None  # The current object (widget) identified by the xpath
        self.accessible_items = None  # List of QAccessibleInterface items
        # corresponding to self.xpath. It's defined here as a class property to
        # prevent GC bugs ("Internal C++ Object Already Deleted"). These seem
        # to be bugs in PySide2 with the QAccessibleInterface.object() method.
        # Perhaps it is not incrementing the reference count properly.

    def current_widget(self):
        """Returns the current widget."""
        return self.curr_object

    @staticmethod
    def verify_enabled(widget, enabled):
        """Ensure a widget is enabled or disabled.

        Args:
            widget (QWidget): The widget to check
            enabled (bool): The expected enabled status
        """
        if widget.isEnabled() != enabled:
            assert f'widget.isEnabled() is {widget.isEnabled()} != {enabled}'

    @staticmethod
    def verify_check_box(widget, checked):
        """Ensure a checkbox is checked or unchecked.

        Args:
            widget (QCheckBox): The widget to check
            checked (bool): The expected check state
        """
        if widget.isChecked() != checked:
            assert f'widget.isChecked() is {widget.isChecked()} != {checked}'

    @staticmethod
    def verify_combo_box_text(widget, combo_box_text):
        """Ensure a combobox has expected text.

        Args:
            widget (QComboBox): The widget to check
            combo_box_text (str): The expected text
        """
        if widget.currentText() != combo_box_text:
            assert f'widget.currentText() is {widget.currentText()} != {combo_box_text}'

    @staticmethod
    def verify_combo_box_index(widget, index):
        """Ensure a combobox has expected index.

        Args:
            widget (QComboBox): The widget to check
            index (int): The expected index
        """
        if widget.currentIndex() != index:
            assert f'widget.currentIndex() is {widget.currentIndex()} != {index}'

    @staticmethod
    def verify_radio_button(widget, checked):
        """Ensure a radio button has expected state.

        Args:
            widget (QRadioButton): The widget to check
            checked (bool): The expected state
        """
        if widget.isChecked() != checked:
            assert f'widget.isChecked() is {widget.isChecked()} != {checked}'

    @staticmethod
    def verify_line_edit(widget, text):
        """Ensure a line edit has expected value.

        Args:
            widget (QLineEdit): The widget to check
            text (str): The expected text
        """
        if widget.text() != text:
            assert f'widget.text() is {widget.text()} != {text}'

    @staticmethod
    def verify_spin_box(widget, value):
        """Ensure a spin box has expected value.

        Args:
            widget (QSpinBox): The widget to check
            value (str): The expected value
        """
        if widget.value() != value:
            assert f'widget.value() is: {widget.value()} != {value}'

    @staticmethod
    def verify_tab_index(widget, tab_index):
        """Ensure a tab widget has expected index.

        Args:
            widget (QTabWidget): The widget to check
            tab_index (int): The expected tab index
        """
        if widget.currentIndex() != tab_index:
            assert f'widget.currentIndex() is: {widget.currentIndex()} != {tab_index}'

    @staticmethod
    def verify_tabs(widget, tabs):
        """Ensure a tab widget tabs have expected names.

        Args:
            widget (QTabWidget): The widget to check
            tabs (list[str]): The expected tab names
        """
        existing_tabs = []
        for tab_index in range(widget.count()):
            existing_tabs.append(widget.tabText(tab_index))
        if existing_tabs != tabs:
            assert f'tabs are: {existing_tabs} != {tabs}'

    @staticmethod
    def verify_list_widget(widget, expected_contents):
        """Ensure a list widget has expected contents.

        Args:
            widget (QListWidget): The widget to check
            expected_contents (list): The expected list contents
        """
        list_contents = GuiTester.list_widget_contents(widget)
        if list_contents != expected_contents:
            assert f'List contents is: {list_contents} != {expected_contents}'

    @staticmethod
    def verify_table_widget_with_file(widget, filename, row_beg=0, col_beg=0, row_end=-1, col_end=-1):
        """Compare a table widget against a baseline file.

        Args:
            widget (QTableWidget): Widget to check
            filename (str): Path to the baseline file
            row_beg (int): Row to begin checking
            col_beg (int): Row to end checking
            row_end (int): Column to start checking
            col_end (int): Column to end checking
        """
        table_str = GuiTester.table_widget_contents_as_string(widget, row_beg, col_beg, row_end, col_end)
        with open(filename, 'r') as f:
            file_str = f.read()
            if table_str != file_str:
                assert f'Table contents is:\n{table_str}!=\n{file_str}'

    @staticmethod
    def verify_table_view_with_file(widget, filename, row_beg=0, col_beg=0, row_end=-1, col_end=-1):
        """Compare a table view against a baseline file.

        Args:
            widget (QTableView): Widget to check
            filename (str): Path to the baseline file
            row_beg (int): Row to begin checking
            col_beg (int): Row to end checking
            row_end (int): Column to start checking
            col_end (int): Column to end checking
        """
        table_str = GuiTester.table_view_contents_as_string(widget, row_beg, col_beg, row_end, col_end)
        with open(filename, 'r') as f:
            file_str = f.read()
            if table_str != file_str:
                assert f'Table contents is:\n{table_str}!=\n{file_str}'

    @staticmethod
    def list_widget_contents(widget):
        """Returns a the contents of a list widget.

        Args:
            widget (QListWidget): The widget to retrieve contents of

        Returns:
            (list): See description
        """
        contents = []
        for i in range(widget.count()):
            item = widget.item(i)
            contents.append(item.text())
        return contents

    @staticmethod
    def table_widget_contents_as_string(table_widget, row_beg, col_beg, row_end, col_end):
        """Returns a the contents of a table widget given a range.

        Args:
            table_widget (QTableWidget): The widget to retrieve contents of
            row_beg (int): Row to begin retrieval
            col_beg (int): Row to end retrieval
            row_end (int): Column to start retrieval
            col_end (int): Column to end retrieval

        Returns:
            (list): See description
        """
        if row_end < 0:
            row_end = table_widget.rowCount()
        if col_end < 0:
            col_end = table_widget.columnCount()
        contents = ''
        for row in range(row_beg, row_end):
            for col in range(col_beg, col_end):
                widget_item = table_widget.item(row, col)
                if widget_item:
                    contents += widget_item.text()
                if col < col_end - 1:
                    contents += '\t'
            contents += '\n'
        return contents

    @staticmethod
    def table_view_contents_as_string(table_view, row_beg, col_beg, row_end, col_end):
        """Returns a the contents of a table view given a range.

        Args:
            table_view (QTableView): The widget to retrieve contents of
            row_beg (int): Row to begin retrieval
            col_beg (int): Row to end retrieval
            row_end (int): Column to start retrieval
            col_end (int): Column to end retrieval

        Returns:
            (list): See description
        """
        if row_end < 0:
            row_end = table_view.model().rowCount()
        if col_end < 0:
            col_end = table_view.model().columnCount()
        contents = ''
        for row in range(row_beg, row_end):
            for col in range(col_beg, col_end):
                data = table_view.model().index(row, col).data()
                contents += str(data)
                if col < col_end - 1:
                    contents += '\t'
            contents += '\n'
        return contents

    @staticmethod
    def launch_script(script_filename, qapp):
        """Runs the script file on the dialog.

        Args:
            script_filename (str): Filepath to script file.
            qapp (QApplication): The Qt app
        """
        if not os.path.isfile(script_filename):
            if not GuiTester._launch_script_saver_dialog(script_filename):
                return

        spec = importlib.util.spec_from_file_location('module.name', script_filename)
        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)
        module.run(qapp)

    @staticmethod
    def _launch_script_saver_dialog(script_filename):
        """Opens a dialog so user can paste script text and save it to a file.

        Args:
            script_filename (str): Filepath to script file.
        """
        dialog = ScriptSaverDialog(None, script_filename)
        if dialog.exec_() == QDialog.Accepted:
            script_text = dialog.ui.edtScriptText.toPlainText()

            # Convert to python if necessary
            if script_text.startswith('//'):
                command_lines = script_text.split('\n')
                script_text = GuiTester.parse_c_sharp_commands(command_lines)

            # Save to file
            with open(script_filename, 'w') as f:
                f.write(script_text)
            return True
        return False

    def get_xpath_object(self, xpath):
        """Gets the object from the xpath.

        Args:
            xpath (str): xpath to widget from script file.
        """
        if xpath == self.xpath:
            return self.curr_object

        identifier_list = self.identifier_list_from_xpath(xpath)
        self.accessible_items = self.accessible_items_from_identifier_list(identifier_list)

        if self.accessible_items[-1].role() == QAccessible.PageTab:  # Special case for tab items
            return self.accessible_items[-2].object()
        else:
            return self.accessible_items[-1].object()

    def left_click(self, xpath):
        """Performs a left click on the widget specified by the xpath.

        Args:
            xpath (str): xpath to widget from script file.
        """
        # print('left_click')
        self.xpath = xpath
        identifier_list = self.identifier_list_from_xpath(xpath)

        # See if this is a combo box list item click (special case)
        if 'Class' in identifier_list[0] and identifier_list[0]['Class'] == 'QComboBoxPrivateContainer':
            self._left_click_combo_box_item(identifier_list[2]['Name'])
            return

        self.accessible_items = self.accessible_items_from_identifier_list(identifier_list)

        # Do the left click
        last_item = self.accessible_items[-1]
        if last_item.role() == QAccessible.PageTab:  # Special case for tab items
            self.curr_object = GuiTester._left_click_tab(self.accessible_items)
        elif last_item.role() == QAccessible.Role.ListItem:
            self.curr_object = GuiTester._left_click_list_item(self.accessible_items)
        else:  # QAccessible.EditableText, QAccessible.Button)
            obj = last_item.object()
            QTest.mouseClick(obj, Qt.LeftButton)
            self.curr_object = last_item.object()  # Save this for later

        return self.curr_object

    def accessible_items_from_identifier_list(self, identifier_list):
        """Returns list of widgets identified by identifier_list.

        Args:
            identifier_list (list[dict]): List of identifier dicts identifying a widget.

        Returns:
            See description.
        """
        accessible_items = []

        # Start by looking at the top level widgets
        top_level_widgets = self.qapp.topLevelWidgets()
        children = []
        for widget in top_level_widgets:
            children.append(QAccessible.queryAccessibleInterface(widget))
        new_item = self._widget_from_identifier(children, identifier_list[0])
        curr_item = new_item
        accessible_items.append(new_item)

        # Continue down
        for identifier in identifier_list[1:]:
            children = self._accessible_children(curr_item)
            new_item = self._widget_from_identifier(children, identifier)
            accessible_items.append(new_item)
            curr_item = new_item
        return accessible_items

    @staticmethod
    def _accessible_children(accessible):
        """Returns a list of a widget's accessible children.

        Args:
            accessible (QWidget): The parent widget

        Returns:
            (list): See description
        """
        children = []
        for i in range(accessible.childCount()):
            child = accessible.child(i)
            children.append(child)
        return children

    def _widget_from_identifier(self, children, identifier):
        """Returns a QWidget given its identifier.

        Args:
            children (list): List of accesible children
            identifier (dict): Widget identifier

        Returns:
            See description
        """
        if 'Name' in identifier and 'Class' in identifier:
            new_item = self._find_child_from_name_and_class_name(children, identifier)
        elif 'Name' in identifier:
            new_item = self._find_child_from_name(children, identifier)
        elif 'AutomationId' in identifier:
            new_item = self._find_child_from_automation_id(children, identifier)
        elif 'Class' in identifier:
            new_item = self._find_child_from_class_name(children, identifier)
        else:
            new_item = None
        return new_item

    @staticmethod
    def _left_click_tab(accessible_items):
        """Performs left click on a tab in a tab control.

        It appears you can't get a tab object to click on from the
         QAccessibleInterface so we go back up the stack and find the QTabBar
         and click on the item as a workaround.

        Args:
            accessible_items(list): List of QAccessibleInterface items.

        Returns:
            The QTabBar
        """
        name = accessible_items[-1].text(QAccessible.Name)
        tab_bar = accessible_items[-2].object()
        tab_idx = None
        for i in range(tab_bar.count()):
            if tab_bar.tabText(i) == name:
                tab_idx = i
                break
        if tab_idx >= 0:
            # Mouse click in the middle of the tab
            tab_rect = tab_bar.tabRect(tab_idx)
            pos = tab_rect.center()
            QTest.mouseClick(tab_bar, Qt.LeftButton, modifier=None, pos=pos)

        return tab_bar

    @staticmethod
    def _left_click_list_item(accessible_items):
        """Performs left click on a list item in a list view or list widget.

        It appears you can't get a list item object to click on from the
         QAccessibleInterface so we go back up the stack and find the QListView
         (or QListWidget) and click on the item as a workaround.

        Args:
            accessible_items(list): List of QAccessibleInterface items.

        Returns:
            The QListView (or QListWidget)
        """
        name = accessible_items[-1].text(QAccessible.Name)
        list_view_or_widget = accessible_items[-2].object()
        model = list_view_or_widget.model()
        item_idx = None
        for i in range(model.rowCount(QModelIndex())):
            if model.index(i, 0).data() == name:
                item_idx = i
                break
        if item_idx >= 0:
            # Mouse click in the middle of the tab
            item_rect = list_view_or_widget.rectForIndex(model.index(item_idx, 0))
            pos = item_rect.center()
            QTest.mouseClick(list_view_or_widget, Qt.LeftButton, modifier=None, pos=pos)

        return list_view_or_widget

    def _left_click_combo_box_item(self, list_item_name):
        """Performs a left click on an item in a combo box.

        Combo boxes are troublesome in that the drop-down list appears to be a
        temporary window that is hard to get to and hard to relate back to the
        combo box it came from. So, as a workaround, we save the combo box that
        was clicked on previously and use it to call setCurrentText.

        Args:
            list_item_name (str): The list item string clicked on.
        """
        if not self.curr_object:
            return
        self.curr_object.setCurrentText(list_item_name)

    def key_click(self, key_string):
        """Performs a key click.

        Args:
            key_string (str): The string.
        """
        if not self.curr_object:
            return

        if key_string.find('Keys.') == 0:
            tokens = key_string.split('+')
            tokens = [token.strip(' ') for token in tokens]

            # In alphabetical order
            if tokens[0] == 'Keys.ArrowLeft':
                QTest.keyClick(self.curr_object, Qt.Key_Left)
            elif tokens[0] == 'Keys.ArrowRight':
                QTest.keyClick(self.curr_object, Qt.Key_Right)
            elif tokens[0] == 'Keys.Backspace':
                QTest.keyClick(self.curr_object, Qt.Key_Backspace)
            elif tokens[0] == 'Keys.Delete':
                QTest.keyClick(self.curr_object, Qt.Key_Delete)
            elif tokens[0] == 'Keys.End':
                QTest.keyClick(self.curr_object, Qt.Key_End)
            elif tokens[0] == 'Keys.Home':
                QTest.keyClick(self.curr_object, Qt.Key_Home)
            elif tokens[0] in ['Keys.LeftControl', 'Keys.RightControl']:
                char = tokens[1].strip('"')
                QTest.keyClick(self.curr_object, char, Qt.ControlModifier)
            elif tokens[0] == 'Keys.Space':
                QTest.keyClick(self.curr_object, Qt.Key_Space)
        else:
            QTest.keyClicks(self.curr_object, key_string)

        # print(f'key_click: {key_string}')

    @staticmethod
    def _find_child_from_name_and_class_name(children, identifier):
        """Returns the QAccessibleInterface child with the given name and class name.

        Args:
            children (list[QAccessible]): A list of QAccessible children.
            identifier (dict): One part of the xpath as a dict.

        Returns:
            See description.
        """
        name = identifier['Name']
        class_name = identifier['Class']
        for child in children:
            if child.text(QAccessible.Name) == name:
                obj = child.object()
                if obj and obj.__class__.__name__ == class_name:
                    return child
        return None

    @staticmethod
    def _find_child_from_name(children, identifier):
        """Returns the QAccessibleInterface child with the given name.

        Args:
            children (list[QAccessible]): A list of QAccessible children.
            identifier (dict): One part of the xpath as a dict.

        Returns:
            See description.
        """
        name = identifier['Name']
        for child in children:
            if child.text(QAccessible.Name) == name:
                return child
        return None

    @staticmethod
    def _find_child_from_class_name(children, identifier):
        """Returns the QAccessibleInterface child with the given class name.

        Args:
            children (list[QAccessible]): A list of QAccessible children.
            identifier (dict): One part of the xpath as a dict.

        Returns:
            See description.
        """
        class_name = identifier['Class']
        for child in children:
            obj = child.object()
            if obj and obj.__class__.__name__ == class_name:
                return child
        return None

    @staticmethod
    def _find_child_from_automation_id(children, identifier):
        """Returns the QAccessibleInterface child with the given automation id.

        The automation id is apparently the object name.

        Args:
            children (list[QAccessible]): A list of QAccessible children.
            identifier (dict): One part of the xpath as a dict.

        Returns:
            See description.
        """
        automation_id = identifier['AutomationId']
        for child in children:
            obj = child.object()
            if obj and obj.objectName() == automation_id:
                return child
        return None

    @staticmethod
    def identifier_list_from_xpath(xpath):
        """Returns the xpath as a list of dicts.

        Args:
            xpath (str): Xpath to a widget.

        Returns:
            See description.
        """
        xpath = xpath.replace('}', '')
        path_parts = xpath.split('{')
        identifier_list = []
        for part in path_parts:
            if not part:
                continue
            object_items = part.split(';')
            identifier = {}
            for item in object_items:
                key_value = item.split('=')
                key = key_value[0]
                value = key_value[1].strip('"')
                if key == 'AutomationId':
                    last_dot_idx = value.rfind('.')
                    if last_dot_idx >= 0:
                        last_dot_idx += 1
                        value = value[last_dot_idx:]
                identifier[key] = value
            identifier_list.append(identifier)
        return identifier_list

    @staticmethod
    def parse_c_sharp_commands(command_lines):
        """Returns the python code equivalent from the CSharp code as one string.

        Args:
            command_lines(list): List of CSharp lines.

        Returns:
            See description
        """
        # boilerplate
        # commands.append('import os')
        # commands.append('from tests import tests_util')
        commands = [
            '# Code generated by GuiTester.parse_c_sharp_commands version 0.0.0',
            'from xms.guipy.testing.gui_tester import GuiTester',
            '',
            '',
            'def run(qapp):',
            '    tester = GuiTester(qapp)',
            '',
        ]

        line_idx = 0
        while line_idx < len(command_lines):
            line = command_lines[line_idx]
            if line.find('// LeftClick') == 0:
                commands.append(f'    #{line[2:]}')
                variable = GuiTester.get_widget_variable_name(line)

                # skip to xpath line
                line_idx += 2
                line = command_lines[line_idx]
                xpath_start = line.find('"') + 1
                xpath = line[xpath_start:-3].replace('\\"', '"')
                xpath = GuiTester.simplify_xpath(xpath)
                commands.append(f"    xpath = '{xpath}'")  # noqa: B028
                commands.append(f'    {variable} = tester.left_click(xpath)')
                commands.append('')
            elif line.find('// KeyboardInput') == 0:
                commands.append(f'    #{line[2:]}')
                # skip to SendKeys lines
                while line.find('SendKeys') < 0:
                    line_idx += 1
                    line = command_lines[line_idx]
                while line.find('SendKeys') >= 0:
                    keys_start = line.find('(') + 1
                    keys = line[keys_start:-2].strip('\"')
                    commands.append(f"    tester.key_click('{keys}')")  # noqa: B028
                    line_idx += 1
                    line = command_lines[line_idx]
                commands.append('')
            line_idx += 1
        # return '\r\n'.join(commands)
        return '\n'.join(commands)

    @staticmethod
    def simplify_xpath(xpath):
        """Returns the CSharp xpath in a shorter form.

        The first couple of items that get us to the dialog are removed
         entirely. Then some of the stuff that we don't need are removed, like
         "/Group" and the '@' symbols. "ClassName" is shortened to "Class".
         Hopefully there aren't symbols like ';' and '{' in the names because
         we use those as separators and we aren't very smart about parsing.

        Args:
            xpath (str): xpath string in CSharp

        Returns:
            (str): See description
        """
        window_idx = xpath.find('Window')
        wpath = xpath[window_idx:]
        path_parts = wpath.split('/')
        parsed_xpath = []
        for part in path_parts:
            splitable_object_items = part.replace(']', '')
            identifier_items = splitable_object_items.split('[')
            identifier_items.pop(0)
            identifier_list = []
            parsed_xpath.append('{')
            for item in identifier_items:
                item = item.replace('@ClassName', '@Class')
                identifier_list.append(item[1:])
            identifier_string = ';'.join(identifier_list)
            parsed_xpath.append(identifier_string)
            parsed_xpath.append('}')
        new_path = ''.join(parsed_xpath)
        return new_path

    @staticmethod
    def get_widget_variable_name(comment_line):
        """Returns a variable name for a widget given the comment line.

        Args:
            comment_line (str): Comment line that comes before every UI event.

        Returns:
            See description.

        """
        # Widget
        widget_idx0 = comment_line.find('on')
        widget_idx0 += 3
        widget_idx1 = comment_line.find(' ', widget_idx0)
        widget = comment_line[widget_idx0:widget_idx1]

        # Widget name
        quote_idx0 = comment_line.find('"')
        quote_idx0 += 1
        quote_idx1 = comment_line.find('"', quote_idx0)
        widget_name = comment_line[quote_idx0:quote_idx1]
        widget_name = widget_name.replace(' ', '')

        # Point
        parens_0 = comment_line.find('(')
        parens_0 += 1
        parens_1 = comment_line.find(')', )
        point = comment_line[parens_0:parens_1]
        point = point.replace(',', '_')

        variable = f'{widget}_{widget_name}_{point}'

        # Replace any illegal characters with underscore
        variable = re.sub('[^a-zA-Z0-9\n]', '_', variable)
        variable = variable.lower()  # Make it lowercase so Pycharm doesn't bark

        return variable

    @staticmethod
    def qWait(msec):  # noqa: N802
        """Pause updating Qt event loop for specified time.

        Args:
            msec (float): The wait interval in milliseconds
        """
        # PySide2 doesn't have qWait for some reason see:
        # https://github.com/pyqtgraph/pyqtgraph/pull/376/commits/8bdc19be75a7552cc0043bf8b5f5e0ee796edda0
        import time
        start = time.time()
        QApplication.processEvents()
        while time.time() < start + msec * 0.001:
            QApplication.processEvents()


def main():
    """Test driver for the GuiTester."""
    import os
    from tests import tests_util
    from tests.files.test_tools_tests.test_dialog import TestDialog
    from xms.guipy.dialogs import dialog_util

    # Convert the CSharp into python
    test_files = tests_util.get_test_files_path()
    csharp_file = os.path.join(test_files, 'test_tools_tests', 'test_dialog_script.cs')
    with open(csharp_file) as f:
        # command_lines = f.readlines()
        command_lines = f.read().splitlines()  # This way removes all the '\n'
    _ = GuiTester.parse_c_sharp_commands(command_lines)

    # Run the test dialog python script
    qapp = dialog_util.ensure_qapplication_exists()
    _ = TestDialog(None)
    python_file = os.path.join(test_files, 'test_tools_tests', 'test_dialog_script.py')
    GuiTester.launch_script(python_file, qapp)


if __name__ == "__main__":
    main()
