"""Event filter for formatting numbers in editable combo box fields for XMS Python dialogs."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QEvent
from PySide2.QtGui import QValidator

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.validators.number_corrector import NumberCorrector  # noqa: AQU103  We're implementing the replacement.
from xms.guipy.validators.qx_double_validator import QxDoubleValidator


class CbxNumberCorrector(NumberCorrector):
    """Formats string fields to be valid doubles for editable QComboBox widgets."""
    def __init__(self, parent=None):
        """Construct the event filter."""
        super().__init__(parent)
        self._editable_indices = []
        self._modified = False
        self.default_value = '0.0'  # Value to revert to if non-numeric text is entered

    def set_editable_items(self, indices):
        """Set which combobox items are editable.

        Args:
            indices (set): Set of the combobox item indices that are editable.
        """
        self._editable_indices = indices

    def eventFilter(self, obj, event):  # noqa: N802
        """Validate text as it is being inputted.

        Args:
            obj (QObject): Qt object to handle event for
            event (QEvent): The Qt event to handle

        Returns:
            (bool): True if the event was handled
        """
        correct_event = event.type() in [QEvent.FocusOut, QEvent.Close]
        if correct_event and obj and obj.currentIndex() in self._editable_indices:
            self._modified = False
            valid = obj.validator()
            is_dbl = isinstance(valid, QxDoubleValidator)
            line_edit_obj = obj.lineEdit()  # Get the combobox's line edit
            curr_text = line_edit_obj.text()
            if not curr_text:  # Default to 0.0 or 0 if empty string
                curr_text = '0.0' if is_dbl else '0'
                line_edit_obj.setText(curr_text)
                self._modified = True

            state, val, ok = valid.validate(line_edit_obj.text(), 0)
            if is_dbl and (state == QValidator.Intermediate or state == QValidator.Invalid):
                self._fix_invalid_double(line_edit_obj, valid)

            if is_dbl:  # add/trim trailing zeros as needed
                self._fix_double_precision_padding(line_edit_obj)
            elif state == QValidator.Intermediate or state == QValidator.Invalid:  # Integers
                self._fix_invalid_integer(line_edit_obj, valid)

            if self._modified:  # Notify connected objects that text changed
                line_edit_obj.setModified(True)
                self.text_corrected.emit()
        return super(NumberCorrector, self).eventFilter(obj, event)

    def _fix_double_precision_padding(self, line_edit_obj):
        """Trim or add trailing zeroes to a floating point string.

        Args:
            line_edit_obj (QLineEdit): The line edit widget containing the text to correct
        """
        current, ok = NumberCorrector.MERICA_LOCALE.toDouble(line_edit_obj.text())
        s = self.format_double(current)
        line_edit_obj.setText(s)
        self._modified = True

    def _fix_invalid_double(self, line_edit_obj, validator):
        """Format a string that is an invalid double.

        Args:
            line_edit_obj (QLineEdit): The line edit widget containing the text to correct
            validator (QValidator): The validator of the line edit widget
        """
        # Change the text to be something valid
        prec = validator.decimals()
        current, ok = NumberCorrector.MERICA_LOCALE.toDouble(line_edit_obj.text())
        if ok:  # Is a valid double, but out of range
            if current < validator.bottom():
                line_edit_obj.setText(f'{validator.bottom():.{prec}f}')
            else:
                line_edit_obj.setText(f'{validator.top():.{prec}f}')
        else:
            line_edit_obj.setText(self.default_value)
        self._modified = True

    def _fix_invalid_integer(self, line_edit_obj, validator):
        """Format a string that is an invalid integer.

        Args:
            line_edit_obj (QLineEdit): The line edit widget containing the text to correct
            validator (QValidator): The validator of the line edit widget
        """
        current, ok = NumberCorrector.MERICA_LOCALE.toInt(line_edit_obj.text())
        if ok:  # Is a valid integer, but out of range
            if current <= validator.bottom():
                line_edit_obj.setText(NumberCorrector.MERICA_LOCALE.toString(validator.bottom()))
            else:
                line_edit_obj.setText(NumberCorrector.MERICA_LOCALE.toString(validator.top()))
        else:
            line_edit_obj.setText(NumberCorrector.MERICA_LOCALE.toString(validator.bottom()))
        self._modified = True
