"""XMS project explorer tree item picker."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QTreeWidget, QTreeWidgetItem, QTreeWidgetItemIterator

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util

# 4. Local modules
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.resources import resources_util

col0 = 0  # Column 0 (to avoid magic number 0's everywhere).


class QxProjectExplorerSelectorWidget(QTreeWidget):
    """A widget showing items from the Project Explorer in a tree view."""
    def __init__(
        self,
        parent=None,
        root_node=None,
        selectable_item_type='',
        previous_selection='',
        override_icon=None,
        show_root=False,
        selectable_xms_types=None,
        allow_multi_select=False,
        allow_change_selection=True,
        selectable_callback=None,
        always_show_check_boxes: bool = True,
        filter_tree: bool = True
    ):
        """Initializes the class.

        Args:
            root_node (TreeNode): Root of the tree to display in the dialog
            selectable_item_type (type): Type of item in project_tree that the user can select. Only these items
                and their ancestors will be shown in the widget.
            parent: Parent window
            previous_selection (str or iterable): UUID of the previous selection. If provided, that item will be
                selected when the dialog appears. Can be an iterable container of previously selected UUIDs (implies
                allow_multi_select=True).
            override_icon (callable): Callable method to provide icons that override the default for a tree item
                type. Method should take a TreeNode and return an icon path if it is overriden, else empty string. Icon
                path must be in whatever format that works in the calling package.
            show_root (bool): If True, root of the tree will be shown
            selectable_xms_types (list): XMS tree item types of selectable items that are not directly supported
                by xmsapi.
            allow_multi_select (bool): If True, multiple tree items may be selected.
            allow_change_selection (bool): If False, the selection cannot be changed (used for display purposes).
            selectable_callback: A callable function that takes a TreeNode and returns True if it is selectable.
            always_show_check_boxes (bool): If False, only shows checkboxes if allow_multi_select is True.
            filter_tree (bool): If False, assumes the calling code has already filtered the tree to desired nodes and
                does not remove anything.
        """
        super().__init__(parent)
        self._root_node = root_node
        self._selectable_item_type = selectable_item_type  # Type of items we want to select.
        # Selectable XMS tree item types that are not directly supported by xmsapi.
        self._selectable_xms_types = selectable_xms_types if selectable_xms_types else []
        self._selectable_callback = selectable_callback
        self._override_icon = override_icon
        self._show_root = show_root
        self._in_on_item_changed = False  # Flag to avoid recursion when changing checked state
        self._allow_multi_select = allow_multi_select
        self._always_show_check_boxes = always_show_check_boxes
        self._allow_change_selection = allow_change_selection
        self._initial_selection = None
        self._icon_paths = {}
        self._filter_tree = filter_tree

        # Setup the tree widget
        self.setHeaderHidden(True)
        self.setColumnCount(1)

        # Signals
        self.itemChanged.connect(self._on_item_changed)

        self.initialize(
            root_node, selectable_item_type, previous_selection, override_icon, show_root, self._selectable_xms_types,
            allow_multi_select, allow_change_selection, selectable_callback, always_show_check_boxes
        )

    def initialize(
        self,
        root_node,
        selectable_item_type,
        previous_selection,
        override_icon,
        show_root,
        selectable_xms_types,
        allow_multi_select,
        allow_change_selection,
        selectable_callback=None,
        always_show_check_boxes: bool = True
    ):
        """Initializes the dialog using the arguments supplied.

        Args:
            root_node (TreeNode): Root of the tree to display in the dialog
            selectable_item_type (str): Type of item in project_tree that the user can select. Only these items
                and their ancestors will be shown in the widget.
            previous_selection (str or iterable): UUID of the previous selection. If provided, that item will be
                selected when the dialog appears. Can be an iterable container of previously selected UUIDs (implies
                allow_multi_select=True).
            override_icon (callable): Callable method to provide icons that override the default for a tree item
                type. Method should take a TreeNode and return an icon path if it is overriden, else empty string. Icon
                path must be in whatever format that works in the calling package.
            show_root (bool): If True, root of the tree will be shown
            selectable_xms_types (list): XMS tree item types of selectable items that are not directly supported
                by xmsapi.
            allow_multi_select (bool): If True, multiple tree items may be selected.
            allow_change_selection (bool): If False, the selection cannot be changed (used for display purposes).
            selectable_callback: A callable function that takes a TreeNode and returns True if it is selectable.
            always_show_check_boxes (bool): If False, only shows checkboxes if allow_multi_select is True.
        """
        self._show_root = show_root
        self._override_icon = override_icon
        self._selectable_item_type = selectable_item_type
        self._selectable_xms_types = selectable_xms_types
        self._selectable_callback = selectable_callback
        self._allow_multi_select = allow_multi_select
        self._always_show_check_boxes = always_show_check_boxes
        self._allow_change_selection = allow_change_selection
        if self._filter_tree:
            self._root_node = tree_util.trim_tree_to_items_of_type(
                root_node, [self._selectable_item_type], self._selectable_xms_types
            )
        else:
            self._root_node = root_node

        self._add_tree_items(self._root_node, self)

        if not previous_selection:
            return  # No known previous selection.

        # If the previous selection is a single UUID, put it in a list. When multi-select is enabled, there may be
        # more than one previously selected item.
        if isinstance(previous_selection, str):
            previous_selection = [previous_selection]

        self._initial_selection = previous_selection

        # Check the previously selected item(s).
        it = QTreeWidgetItemIterator(self)
        while it.value():
            item = it.value()
            if self._is_initial_selection(item, previous_selection):
                if not allow_change_selection:
                    # Don't show any message boxes here if we're not allowing users to change the selection.
                    self._in_on_item_changed = True
                self._set_item_selected(item, True)
                if not allow_change_selection:
                    self._in_on_item_changed = False
                if not self._allow_multi_select:
                    return  # Only one item may be selected at a time.
            it += 1

    def _is_initial_selection(self, item: QTreeWidgetItem, previous_selection) -> bool:
        """Returns True if the item matches the initial.

        Args:
            item (QTreeWidgetItem): A tree item.
            previous_selection: String or list of strings.

        Returns:
            (bool): See description.
        """
        return self._item_is_selectable(item) and item.data(col0, Qt.UserRole) in previous_selection

    def _item_is_selectable(self, item: QTreeWidgetItem) -> bool:
        """Returns True if the item is selectable.

        Args:
            item (QTreeWidgetItem): A tree item.

        Returns:
            (bool): See description.
        """
        if self._always_show_check_boxes or self._allow_multi_select:
            return item and item.flags() & Qt.ItemIsUserCheckable
        else:
            return item and item.flags() & Qt.ItemIsSelectable

    def _set_item_selected(self, item: QTreeWidgetItem, selected: bool) -> None:
        """Sets the item to be selected or not.

        Args:
            item (QTreeWidgetItem): A tree item.
            selected (bool): True if it should be selected, else False.
        """
        if item:
            if self._always_show_check_boxes or self._allow_multi_select:
                item.setCheckState(col0, Qt.Checked if selected else Qt.Unchecked)
            else:
                item.setSelected(selected)

    def _item_is_selected(self, item: QTreeWidgetItem) -> bool:
        """Returns True if the item is selected.

        Args:
            item (QTreeWidgetItem): A tree item.

        Returns:
            (bool): See description.
        """
        if self._always_show_check_boxes or self._allow_multi_select:
            return item and item.flags() & Qt.ItemIsUserCheckable and item.checkState(col0) == Qt.Checked
        else:
            return item and item.flags() & Qt.ItemIsSelectable and item.isSelected()

    def _unselect_other_items(self, except_item):
        """Unchecks all checkable items except for except_item.

        Called from self._on_item_changed
        Args:
            except_item (QTreeWidgetItem): Item to leave unchanged.s
        """
        it = QTreeWidgetItemIterator(self)
        while it.value():
            item = it.value()
            if self._item_is_selectable(item) and item is not except_item:
                self._set_item_selected(item, False)
            it += 1

    def _on_item_changed(self, item, column):
        """Slot called when itemChanged signal sent. Unchecks other items.

        Args:
            item (QTreeWidgetItem): The item that was changed.
            column (int): The column.
        """
        if self._in_on_item_changed:
            return  # We're already in this function. Return to stop recursion.
        self._in_on_item_changed = True
        # If we're not allowing users to change the selection.
        if not self._allow_change_selection:
            show_message = False
            if self._initial_selection and item.data(col0, Qt.UserRole) not in self._initial_selection:
                self._set_item_selected(item, False)
                show_message = True
            if self._initial_selection and item.data(col0, Qt.UserRole) in self._initial_selection:
                self._set_item_selected(item, True)
                show_message = True
            if show_message:
                app_name = os.environ.get('XMS_PYTHON_APP_NAME', '')
                msg = 'Cannot change the selected dataset. Only the time step can be modified.'
                message_with_ok(self, message=msg, app_name=app_name, icon='Information', win_icon=self.windowIcon())
            self._in_on_item_changed = False
            return

        item_selected = self._item_is_selected(item)
        if item_selected and not self._allow_multi_select:
            # If the item was selected and multi-select is disabled, unselect all other items (act like a radio group).
            self._unselect_other_items(item)
        self._in_on_item_changed = False

    def _set_icon(self, tree_node, widget_item):
        """Sets the icon for the item.

        Args:
            tree_node (TreeNode): Project explorer tree item to set icon for
            widget_item (QTreeWidgetItem): The item widget

        Returns:
            (str): The icon path used to set the icon. Only needed for testing
        """
        icon_path = None
        if self._override_icon:  # First check if icon for this type is being overriden
            icon_path = self._override_icon(tree_node)
        if not icon_path:  # Second check if icon has been defined for the XMS type name
            icon_path = resources_util.get_tree_icon_from_xms_typename(tree_node)
        if not icon_path:  # Default to using the xmsapi C++ type to find something close.
            icon_path = self._find_icon_by_ctype(tree_node)
        if icon_path:
            widget_item.setIcon(col0, QIcon(icon_path))
        return icon_path  # This is just for testing

    def _find_icon_by_ctype(self, tree_node):
        """If all else fails, find a tree item icon based on the xmsapi C++ type.

        Args:
            tree_node (TreeNode): Project explorer tree item to set icon for
        Returns:
            str: Path to the icon if to use
        """
        # Fall through for types that haven't been explicitly specified. Also gives us something for pointer items,
        # even if they don't quite match XMS.
        item_type = type(tree_node.data)
        icon_path = ':resources/icons/folder.svg'  # Default to folder icon
        if item_type == xmd.UGridItem:
            icon_path = ':resources/icons/UGrid_Module_Icon.svg'
        elif item_type == xmd.DatasetItem:
            if tree_node.num_components == 1:  # Scalar
                icon_path = ':resources/icons/dataset_cells_active.svg'
                if tree_node.data_location == 'NODE':  # Node-based scalar
                    icon_path = ':resources/icons/dataset_points_active.svg'
            else:  # Vector
                icon_path = ':/resources/icons/dataset_vector_active.svg'
        elif item_type == xmd.SimulationItem:
            icon_path = ':resources/icons/simulation.svg'
        elif item_type == xmd.CoverageItem:
            icon_path = ':resources/icons/coverage_active.svg'
        elif item_type == xmd.ComponentItem:
            icon_path = ':resources/icons/component.svg'
            # Check if a specific icon has been passed in for this component type.
            comp_type = f'{item_type}#{tree_node.model_name}#{tree_node.unique_name}'
            if comp_type in self._icon_paths:
                icon_path = self._icon_paths[comp_type]
        return icon_path

    def _add_tree_items(self, tree_data, parent):
        """Recursively adds items to the tree.

        Args:
            tree_data (TreeNode): Root of the project explorer tree to add items to
            parent: The parent QTreeWidgetItem. The first time it should be self (QTreeWidget).
        """
        if not tree_data:
            return

        if self._show_root and parent is self:
            item = QTreeWidgetItem(parent)
            self.addTopLevelItem(item)
            self._add_single_tree_item(item, tree_data)
            parent = item

        for child in tree_data.children:
            item = QTreeWidgetItem(parent)
            if not self._show_root and parent is self:
                self.addTopLevelItem(item)

            self._add_single_tree_item(item, child)

            # Recurse on children
            self._add_tree_items(child, item)

    def _add_single_tree_item(self, item, tree_node):
        """Add a tree item node to the tree view.

        Args:
            item (QTreeWidgetItem): The Qt tree widget item to add
            tree_node (TreeNode): The tree item node to add
        """
        item.setText(col0, tree_node.name)
        item.setFlags(item.flags() & ~(Qt.ItemIsSelectable | Qt.ItemIsUserCheckable))  # Make nothing selectable
        self._set_icon(tree_node, item)
        self.expandItem(item)

        # If selectable, set flags and store it's uuid and type
        if self._selectable_callback:
            types_match = self._selectable_callback(tree_node)
        else:
            types_match = type(tree_node.data) is self._selectable_item_type
            types_match = types_match or tree_node.item_typename in self._selectable_xms_types
        if not tree_node.is_ptr_item and types_match:
            item.setData(col0, Qt.UserRole, tree_node.uuid)  # Store uuid
            if self._always_show_check_boxes or self._allow_multi_select:
                item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
                item.setCheckState(col0, Qt.Unchecked)
            else:
                item.setFlags(item.flags() | Qt.ItemIsSelectable)
                item.setSelected(False)

    def get_selected_item_uuid(self):
        """Returns the selected item's uuid or None if nothing selected.

        If multi-select is enabled, return value is a list of the selected tree items' UUIDs. Empty list if no items
        selected.

        Returns:
            See description.
        """
        # Ensure return value is a list if multi-select is enabled and a single str (or None) when disabled.
        selected_uuids = [] if self._allow_multi_select else None

        it = QTreeWidgetItemIterator(self)
        while it.value():
            item = it.value()
            if self._item_is_selected(item):
                data = item.data(col0, Qt.UserRole)
                if self._allow_multi_select:  # If more than one item can be selected, iterate the whole tree.
                    selected_uuids.append(data)
                else:  # If only one item can be selected, we are done.
                    return data
            it += 1

        return selected_uuids

    def has_selectable_items(self):
        """Return True if trimmed tree has items."""
        return self._root_node is not None
