"""QTableWidget implementation."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QKeySequence
from PySide2.QtWidgets import QApplication, QComboBox, QLineEdit, QSpinBox, QTableWidget

# 3. Aquaveo modules

# 4. Local modules


class QxTableWidget(QTableWidget):
    """Implementation of QTableWidget for use in XMS Python packages."""
    def __init__(self, parent=None):
        """Construct the widget."""
        super().__init__(parent)

    def keyPressEvent(self, event):  # noqa: N802
        """To handle copy, paste, insert and delete.

        From https://www.walletfox.com/course/qtableviewcopypaste.php

        Args:
            event (PySide2.QtGui.QKeyEvent): The type of user event to handle
        """
        selected_rows = self.selectionModel().selectedRows()
        # at least one entire row selected
        handled = False
        if selected_rows:
            if event.key() == Qt.Key_Insert:
                self.model().insertRows(selected_rows[0].row(), len(selected_rows))
                handled = True
            elif event.key() == Qt.Key_Delete:
                self.model().removeRows(selected_rows[0].row(), len(selected_rows))
                handled = True

        # at least one cell selected
        if not handled and self.selectedIndexes():
            if event.key() == Qt.Key_Delete:
                selected_indexes = self.selectedIndexes()
                for index in selected_indexes:
                    self.model().setData(index, '')

            elif event.matches(QKeySequence.Copy):
                self.on_copy()

            elif event.matches(QKeySequence.Paste):
                self.on_paste()
            else:
                QTableWidget.keyPressEvent(self, event)

    def on_paste(self):
        """Called on user paste event."""
        text = QApplication.clipboard().text()
        row_contents = list(filter(None, text.split('\n')))
        init_index = self.selectedIndexes()[0]
        init_row = init_index.row()
        init_col = init_index.column()
        for i in range(len(row_contents)):
            column_contents = row_contents[i].split('\t')
            for j in range(len(column_contents)):
                self.set_value(init_row + i, init_col + j, column_contents[j])

    def on_copy(self):
        """Called on user copy event."""
        text = ''
        tab = '\t'
        # For some reason the following crashes so we do it one at a time
        # selection_range = self.selectionModel().selection().first()
        selection_model = self.selectionModel()
        selection = selection_model.selection()
        selection_range = selection.first()
        top = selection_range.top()
        bottom = selection_range.bottom()
        left = selection_range.left()
        right = selection_range.right()
        for i in range(top, bottom + 1):
            row_contents = []
            for j in range(left, right + 1):
                # row_contents.append(self.model().index(i, j).data())
                row_contents.append(self.get_value(i, j))
            text = text + tab.join(row_contents) + '\n'
        QApplication.clipboard().setText(text)

    def get_value(self, row, column):
        """Retrieve a value for a cell in the table."""
        widget = self.cellWidget(row, column)
        if widget:
            if type(widget) is QComboBox:
                return widget.currentText()
            elif type(widget) is QLineEdit:
                return widget.text()
            elif type(widget) is QSpinBox:
                return str(widget.value())
            else:
                raise AssertionError()
        else:
            return self.item(row, column).data(Qt.DisplayRole)

    def set_value(self, row, column, value):
        """Set a value for a cell in the table."""
        widget = self.cellWidget(row, column)
        if widget:
            if type(widget) is QComboBox:
                widget.setCurrentText(str(value))
            elif type(widget) is QLineEdit:
                widget.setText(str(value))
            elif type(widget) is QSpinBox:
                widget.setValue(int(value))
            else:
                raise AssertionError()
        else:
            self.item(row, column).setData(Qt.EditRole, value)
