"""Generates a mesh geometry."""
# 1. Standard python modules
import datetime
import time

# 2. Third party modules
import h5py

# 3. Aquaveo modules

# 4. Local modules
from xms.hecras._TextFormatter import TextFormatter
from xms.hecras._TopologyProcessor import TopologyProcessor


class MeshGenerator:
    """Mesh Generator."""

    groupattributes = [['Version', '1.0'],
                       ['Data Date', None],
                       ['Extents', None, 'float64'],
                       ['Cell Maximum Index', None, 'int32'],
                       ['Cell Maximum Size', None, 'float32'],
                       ['Cell Minimum Size', None, 'float32'],
                       ['Cell Average Size', None, 'float32']]

    rootattributes = [['Extents', None, 'float64'],
                      ['Geometry Time', None],
                      ['Title', 'Sample1'],
                      ['Version', '1.0.1']]

    datasetattributes = [['Cells Center Coordinate', 'float64', [['Row', 'Cell'], ['Column', ['X', 'Y']]]],
                         ['Cells Face and Orientation Info', 'int32',
                          [['Row', 'Cell'], ['Column', ['Starting Index', 'Count']]]],
                         ['Cells Face and Orientation Values', 'int32',
                          [['Row', 'row'], ['Column', ['Face Index', 'Orientation']]]],
                         ['Cells FacePoint Indexes', 'int32', [['Row', 'Cell'], ['Column', ['Face Point Indexes']]]],
                         ['FacePoints Cell Index Values', 'int32', [['Row', 'row'], ['Column', ['Cell Index']]]],
                         ['FacePoints Cell Info', 'int32',
                          [['Row', 'Face Point'], ['Column', ['Starting Index', 'Count']]]],
                         ['FacePoints Coordinate', 'float64', [['Row', 'Face Point'], ['Column', ['X', 'Y']]]],
                         ['FacePoints Face and Orientation Info', 'int32',
                          [['Row', 'Face Point'], ['Column', ['Start Index', 'Count']]]],
                         ['FacePoints Face and Orientation Values', 'int32',
                          [['Row', 'row'], ['Column', ['Face Index', 'Orientation']]]],
                         ['FacePoints Is Perimeter', 'int32', [['Row', 'Face Point'], ['Column', ['Is On Perimeter']]]],
                         ['Faces Cell Indexes', 'int32', [['Row', 'Face'], ['Column', ['Cell 0', 'Cell 1']]]],
                         ['Faces FacePoint Indexes', 'int32',
                          [['Row', 'Face'], ['Column', ['Face Point A', 'Face Point B']]]],
                         ['Faces NormalUnitVector and Length', 'float32',
                          [['Row', 'Face'], ['Column', ['X Component', 'Y Component', 'Face Length']]]],
                         ['Faces Perimeter Info', 'int32',
                          [['Row', 'Face'], ['Column', ['Start Index', 'Count']]]],
                         ['Faces Perimeter Values', 'float64', [['Row', 'row'], ['Column', ['X', 'Y']]]],
                         ['Perimeter', 'float64', [['Row', 'Points'], ['Column', ['X', 'Y']]]]]

    targetpath = None       # Where the files are written
    rawpolygons = None
    nodes = None
    units = None
    flowareaname = None
    bcfilename = None
    units = None        # Units in the geom file
    simunits = None     # Units indicated in the sim file
    topo = None
    extents = None
    datemesh = None
    lookup = None
    starttime = None
    endtime = None
    timestep = None
    reportinterval = None

    def __init__(self, rawpolygons, nodes, units, targetpath, basename, flowareaname="Area2D"):
        """Constructor.

        Args:
            rawpolygons (list): Polygons of the geometry
            nodes (list): Locations of the geometry points
            units (str): The geometry's horizontal projection units
            targetpath (str): Folder to write file in
            basename (str): Name of file to write
            flowareaname (str): Name of the flow area
        """
        self.targetpath = targetpath
        self.flowareaname = flowareaname
        self.rawpolygons = rawpolygons
        self.nodes = nodes
        self.units = units
        self.filenamemesh = basename

    def process(self):
        """Formatted Print function for generating mesh."""
        lasttime = time.time()
        print("===============")
        print("Generating mesh")
        print("===============")
        self.topo = TopologyProcessor(self.rawpolygons, self.nodes, self.units)
        self.topo.process()
        self.calculate_extents()
        self.calculate_date()
        self.calculate_cells_info()
        self.populate_lookup()
        self.crea_hdf_file()
        self.crea_ggx_file()
        self.crea_prj_file()
        print("==============")
        print("Mesh generated")
        print("==============")
        newtime = time.time()
        print(newtime - lasttime)

    def calculate_cells_info(self):
        """Calculates maximum index, max area, minimum area, and average area."""
        maxindex = len(self.topo.centers)
        maxarea = max(self.topo.cells_areas)
        minarea = min(self.topo.cells_areas)
        avgarea = sum(self.topo.cells_areas) / float(len(self.topo.cells_areas))
        self.groupattributes[3][1] = maxindex
        self.groupattributes[4][1] = maxarea
        self.groupattributes[5][1] = minarea
        self.groupattributes[6][1] = avgarea

    def calculate_extents(self):
        """Calculates change in x, change in y, and the extents of the perimeter."""
        perimeter = self.topo.perimeter
        xmax = perimeter[0][0]
        xmin = xmax
        ymax = perimeter[0][1]
        ymin = ymax
        for coords in perimeter:
            if coords[0] > xmax:
                xmax = coords[0]
            if coords[0] < xmin:
                xmin = coords[0]
            if coords[1] > ymax:
                ymax = coords[1]
            if coords[1] < ymin:
                ymin = coords[1]
        # Make extents a 10% larger
        deltax = xmax - xmin
        deltay = ymax - ymin
        extents = [xmin - 0.1 * deltax, xmax + 0.1 * deltax, ymin - 0.1 * deltay, ymax + 0.1 * deltay]
        self.groupattributes[2][1] = extents
        self.rootattributes[0][1] = extents
        self.extents = extents

    def calculate_date(self):
        """Calculates and assigns the current date.

        If the gXX and de hdf files don't have the same date, RAS2D wouldn't load the hdf.
        """
        now = datetime.datetime.now()
        datemesh = now.strftime('%d%b%Y %H:%M:00')
        # ^It requires to have at least seconds rounded to 10. Wouldn't work with 10:25:37,
        # for example, but it would with 10:25:30.
        self.groupattributes[1][1] = datemesh
        self.rootattributes[1][1] = datemesh
        self.datemesh = datemesh

    def crea_ggx_file(self):
        """Creates a ggx file."""
        # Header
        tfm = TextFormatter()
        extents = f'Viewing Rectangle= {self.extents[0]}, {self.extents[1]}, {self.extents[3]}, {self.extents[2]}'
        lines = ["Geom Title=Grid1", "Program Version=5.00", extents]
        # Boundary
        centerx = (self.extents[0] + self.extents[1]) * 0.5
        centery = (self.extents[2] + self.extents[3]) * 0.5
        lines.append("Storage Area=" + self.flowareaname + ", " + str(centerx) + ", " + str(centery))
        lines.append("Storage Area Surface Line= " + str(len(self.topo.perimeter)))
        addlines = tfm.create_text_in_columns(1, 16, self.topo.perimeter)
        lines = lines + addlines
        # Filling
        lines.append("Storage Area Type= 0 ")
        lines.append("Storage Area Area=")
        lines.append("Storage Area Min Elev=")
        lines.append("Storage Area Is2D=-1")
        lines.append("Storage Area Point Generation Data=0,0,1500,1500")
        # Centers
        centersqty = len(self.topo.realcenters)
        lines.append("Storage Area 2D Points= " + str(centersqty))
        points = []
        for realcenter in self.topo.realcenters:
            points.append(realcenter[1])
        addlines = tfm.create_text_in_columns(2, 16, points)
        lines = lines + addlines
        # Filling
        lines.append("Storage Area 2D PointsPerimeterTime=" + self.datemesh)
        lines.append("Storage Area Mannings=0.03")
        lines.append("2D Cell Volume Filter Tolerance=0.01")
        lines.append("2D Face Profile Filter Tolerance=0.01")
        lines.append("2D Face Area Elevation Profile Filter Tolerance=0.01")
        lines.append("2D Face Area Elevation Conveyance Ratio=0.02")
        # Boundary Conditions
        pass
        # Tail
        lines.append("LCMann Time=Dec/30/1899 00:00:00")
        lines.append("LCMann Region Time=Dec/30/1899 00:00:00")
        lines.append("LCMann Table=0")
        lines.append("Chan Stop Cuts=-1")
        lines.append("Use User Specified Reach Order=0")
        lines.append("GIS Ratio Cuts To Invert=-1")
        lines.append("GIS Limit At Bridges=0")
        lines.append("Composite Channel Slope=5")
        # Write file
        f = open(self.targetpath + self.filenamemesh + ".g01", "w")
        for line in lines:
            f.write(line + "\n")
        f.close()

    def crea_prj_file(self):
        """Creates a prj file."""
        lines = ["Proj Title=" + self.filenamemesh, "Current Geom=g01", self.topo.units, "Geom File=g01"]
        f = open(self.targetpath + self.filenamemesh + ".prj", "w")
        for line in lines:
            # print line
            f.write(line + "\n")
        f.close()

    def populate_lookup(self):
        """Populates self.lookup with data."""
        lookup = [['Cells Center Coordinate', self.topo.cells_center_coordinate, True],
                  ['Cells Face and Orientation Info', self.topo.cells_face_and_orientation_info, True],
                  ['Cells Face and Orientation Values', self.topo.cells_face_and_orientation_values, True],
                  ['Cells FacePoint Indexes', self.topo.cells_face_point_indexes, True],
                  ['FacePoints Cell Index Values', self.topo.face_points_cell_index_values, False],
                  ['FacePoints Cell Info', self.topo.face_points_cell_info, True],
                  ['FacePoints Coordinate', self.topo.face_points_coordinate, True],
                  ['FacePoints Face and Orientation Info', self.topo.face_points_face_and_orientation_info, True],
                  ['FacePoints Face and Orientation Values', self.topo.face_points_face_and_orientation_values, True],
                  ['FacePoints Is Perimeter', self.topo.face_points_is_perimeter, False],
                  ['Faces Cell Indexes', self.topo.faces_cell_indexes, True],
                  ['Faces FacePoint Indexes', self.topo.faces_face_point_indexes, True],
                  ['Faces NormalUnitVector and Length', self.topo.faces_normal_unit_vector_and_length, True],
                  ['Faces Perimeter Info', self.topo.faces_perimeter_info, True],
                  ['Faces Perimeter Values', self.topo.faces_perimeter_values, True],
                  ['Perimeter', self.topo.perimeter, True]]
        self.lookup = lookup

    def crea_hdf_file(self):
        """Creates an hdf file."""
        f = h5py.File(self.targetpath + self.filenamemesh + ".g01" + ".hdf", 'w')
        # Create the root group
        mgrp = f.create_group("Geometry/")
        # Populate attributes for the root group
        for groupatt in self.rootattributes:
            name = groupatt[0]
            att = groupatt[1]
            if len(groupatt) == 3:
                dtypeatt = groupatt[2]
                mgrp.attrs.create(name, att, dtype=dtypeatt)
            else:
                dtype = "S" + str(len(att))  # http://docs.h5py.org/en/latest/strings.html#
                mgrp.attrs.create(name, att, dtype=dtype)
                # mgrp.attrs.create(name, att)
        # Create group for geometries
        fgrp = f.create_group("Geometry/2D Flow Areas/" + self.flowareaname)
        # Populate attributes for the group
        for groupatt in self.groupattributes:
            name = groupatt[0]
            att = groupatt[1]
            if len(groupatt) == 3:
                dtypeatt = groupatt[2]
                fgrp.attrs.create(name, att, dtype=dtypeatt)
            else:
                dtype = "S" + str(len(att))  # http://docs.h5py.org/en/latest/strings.html#
                fgrp.attrs.create(name, att, dtype=dtype)
        # Create the other required group
        _ = f.create_group("Geometry/Structures/")
        # Create the datasets
        for dataset in self.datasetattributes:
            name = dataset[0]
            dtype = dataset[1]
            dsetatts = dataset[2]
            morethanonedim = False
            for elem in self.lookup:
                if elem[0] == name:
                    cval = elem[1]
                    morethanonedim = elem[2]
            chunks1 = len(cval)
            if morethanonedim:      # It has more than one dimension
                chunks2 = len(cval[0])
                chunks = (chunks1, chunks2)
            else:
                chunks = (chunks1,)
            newdset = fgrp.create_dataset(name, data=cval, dtype=dtype, chunks=chunks)
            # compression = "gzip", compression_opts = 1)#, chunks = dset.chunks,  \
            # maxshape = dset.maxshape, fillvalue = "nan")

            # newdset = fgrp.create_dataset(name, data = cval)
            # print(name, dtype)
            # Populate the attributes
            for dsetatt in dsetatts:
                name = dsetatt[0]
                att = dsetatt[1]
                if len(dsetatt) == 3:
                    dtypeatt = dsetatt[2]
                    newdset.attrs.create(name, att, dtype=dtypeatt)
                else:
                    dtype = "S" + str(len(att))     # http://docs.h5py.org/en/latest/strings.html#
                    newdset.attrs.create(name, att, dtype=dtype)
        f.close()
