"""TextEditorComponent class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import MenuItem
from xms.core.filesystem import filesystem
from xms.guipy import file_io_util
from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.hgs.components import dmi_util
from xms.hgs.components.hgs_component_base import HgsComponentBase
from xms.hgs.gui import gui_util


class TextEditorComponent(HgsComponentBase):
    """A Dynamic Model Interface (DMI) component for a HydroGeoSphere solution text file viewer component."""
    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file (str): The main file associated with this component.
        """
        super().__init__(main_file)
        cards = file_io_util.read_json_file(Path(main_file))
        self.sim_uuid = cards.get('SIM_UUID', '')
        self.data_file = dmi_util.find_solution_file(
            cards.get('SOLUTION_FILE_FULL', ''), cards.get('SOLUTION_FILE_RELATIVE', ''), main_file
        )

    def get_project_explorer_menus(self, main_file_list: list[str]):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (list of str): A list of the main files of the selected components of this type.

        Returns:
            menu_items (list of xmsapi.dmi.MenuItem): A list of menus and menu items to be shown. Note
            that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        if not main_file_list or len(main_file_list) > 1:
            return []  # Multi-select or nothing selected

        menu_list: list['MenuItem | None'] = []  # None is interpreted as a spacer menu item
        self._add_tree_menu_item('Open...', '_open_dialog', menu_list)
        return menu_list

    def _open_dialog(self, query, params, win_cont):  # pragma no cover - my test for this hangs on the CI mysteriously
        """Opens the package dialog.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        if not self.data_file or not self.data_file.is_file():
            message_box.message_with_ok(parent=win_cont, message='Could not find the solution file.', app_name='GMS')
            return [], []
        gui_util.open_file_in_default_app_or_notepad(self.data_file, win_cont)
        return [], []

    def save_to_location(self, new_path: str, save_type: str):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        new_main_file, messages, actions = super().save_to_location(new_path, save_type)

        new_main_filepath = Path(new_main_file)
        main_filepath = Path(self.main_file)
        if new_main_filepath.resolve() == main_filepath.resolve():
            return new_main_file, messages, actions

        # Fix the paths and rewrite the file
        rewrite = False
        cards = file_io_util.read_json_file(new_main_filepath)
        if cards:
            relative_path = cards.get('SOLUTION_FILE_RELATIVE', '')
            if relative_path:
                rewrite = True
                new_full_path = filesystem.resolve_relative_path(str(main_filepath.parent), relative_path)
                cards['SOLUTION_FILE_FULL'] = new_full_path
                cards['SOLUTION_FILE_RELATIVE'] = filesystem.compute_relative_path(new_path, new_full_path)

        if rewrite:
            file_io_util.write_json_file(cards, new_main_filepath)

        return str(new_main_filepath), messages, actions
