"""Utility functions used by other modules."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import collections.abc
import os
from typing import Sequence

# 2. Third party modules
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.api.tree import TreeNode
from xms.guipy.resources import resources_util

# 4. Local modules

nodata = -9999999
"""A key value meaning the data should be ignored."""

max_int32 = 2147483647  # 2^31 - 1
"""The maximum 32 bit integer value"""

app_name = 'GMS'
"""Name of the XMS app"""


def get_app_icon():
    """Returns the application (GMS) icon."""
    app_icon = QIcon(resources_util.get_resource_path(':/resources/icons/gms.ico'))
    return app_icon


def get_xms_name():
    """Returns the name of the XMS app.

    Returns:
        (str): See description.
    """
    return os.environ.get('XMS_PYTHON_APP_NAME')


def get_xms_version():
    """Returns the XMS version.

    Returns:
        (str): See description.
    """
    return os.environ.get('XMS_PYTHON_APP_VERSION')


def is_non_string_sequence(obj):
    """Returns True if the object is a list, tuple, or similar, and false if it is a string or anything else.

    See https://stackoverflow.com/questions/1835018/how-to-check-if-an-object-is-a-list-or-tuple-but-not-string

    Args:
        obj: Some object.

    Returns:
        (bool): See description.
    """
    return isinstance(obj, collections.abc.Sequence) and not isinstance(obj, str)


def get_dict_value(data_dict, key_path: Sequence[str], default=None):
    """Returns the value of the variable or None if it isn't found.

    See https://stackoverflow.com/questions/31033549/nested-dictionary-value-from-key-path

    Args:
        data_dict: The data dict.
        key_path (Sequence[str]): List-like path to the value in the dict.
        default: Default value to return if not found.
    """
    rv = data_dict
    for key in key_path:
        rv = rv.get(key, None)
        if not rv:
            break
    return rv if rv else default


def set_dict_value(data_dict, key_path, value) -> None:
    """Sets the value in data dict.

    See https://stackoverflow.com/questions/17154393/multiple-levels-of-keys-and-values-in-python

    Args:
        data_dict: The data dict.
        key_path: String or list-like path of keys to the value in the dict.
        value: The value.
    """
    if is_non_string_sequence(key_path):
        rv = data_dict
        for key in key_path[:-1]:
            if key not in rv or rv[key] is None:
                rv[key] = {}
            rv = rv[key]
        rv[key_path[-1]] = value
    else:
        data_dict[key_path] = value


def filter_tree(start_node, condition):
    """Returns a tree filtered by a conditional method.

    tree_util.filter_project_explorer sucks so I wrote this.

    Args:
        start_node (TreeNode): Root of project explorer tree to filter
        condition: Method to apply to tree items. Method should take a TreeNode and return a bool. If True, the item
         and its ancestors will be included in the trimmed tree.

    Returns:
        The root of the new tree.
    """
    if not start_node:
        return None

    children = start_node.children
    if not children:  # Terminal tree item
        return None

    root = TreeNode(other=start_node)  # Create a copy of the root tree item node.
    root.children = []  # Clear children in copy of root item. Want to filter down to selectable types.
    for child in children:
        childs_tree = filter_tree(child, condition)
        if childs_tree:
            root.add_child(childs_tree)
        elif condition(child):
            root.add_child(child)

    root_matches = condition(root)
    return_root = root.children or root_matches
    return root if return_root else None
