"""ImportAh2FileTool class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import Argument, IoDirection, Tool
from xms.tool_core.grid_argument import GridArgument

# 4. Local modules
from xms.hgs.tools import ah2_file_reader

ARG_INPUT_FILE = 0
ARG_OUTPUT_GRID = 1


class ImportAh2FileTool(Tool):
    """Tool to import a AH2 formatted file, which is almost identical to a GMS TIN file."""
    def __init__(self) -> None:
        """Initializes the class."""
        super().__init__(name='Import AH2 File')

        # For testing
        import os
        os.environ['XMSTOOL_GUI_TESTING'] = 'YES'

    def initial_arguments(self) -> list[type[Argument]]:
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.file_argument(name='ah2_file', description='AH2 file'),
            self.grid_argument(name='output_grid', description='Output grid', io_direction=IoDirection.OUTPUT)
        ]
        return arguments

    def run(self, arguments: list[GridArgument]) -> None:
        """Override to run the tool.

        Args:
            arguments (list[GridArgument]): The tool arguments.
        """
        ah2_path = arguments[ARG_INPUT_FILE].text_value
        new_cogrid = ah2_file_reader.read(ah2_path, self.logger)
        self.set_output_grid(new_cogrid, arguments[ARG_OUTPUT_GRID])
