"""CoverageData class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.bases.visible_coverage_component_base_data import VisibleCoverageComponentBaseData
from xms.gmi.data.generic_model import GenericModel
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData

# 4. Local modules


class CoverageData(CoverageBaseData, VisibleCoverageComponentBaseData):
    """Manages data file for a feature coverage component."""
    def __init__(self, main_file: str | Path):
        """
        Initialize the data class.

        Args:
            main_file: The netcdf file (with path) associated with this instance data. Probably the owning
                       component's main file. If not provided, a default one will be chosen.
        """
        super().__init__(main_file)

        self.info.attrs['FILE_TYPE'] = 'HYDROAS_COV'

    @property
    def generic_model(self) -> GenericModel:
        """
        The generic model.

        - This only gets the model's template. Values are not included.
        - This is mainly useful for models with multiple or dynamic templates. Derived models should typically just have
          a function that makes their model template and always use that to get it.
        """
        definitions = self.info.attrs.get('GENERIC_MODEL_TEMPLATE', None)
        generic_model = GenericModel(definitions=definitions)
        return generic_model

    @generic_model.setter
    def generic_model(self, generic_model: GenericModel):
        """
        The generic model.

        This only sets the model's template. Values are not included.
        """
        self.info.attrs['GENERIC_MODEL_TEMPLATE'] = generic_model.to_template().to_string()

    def to_text_file(self, model: GenericModel, file: TextIO):
        """
        Export the data manager's content to a text file in a format suitable for comparing to test baselines.

        Args:
            model: Model describing the data in the manager.
            file: Where to export to.
        """
        file.write('template\n')
        file.write(self.generic_model.to_pretty_string())
        file.write('\n\n')
        super().to_text_file(self.generic_model, file)
