"""Class to fix a .2dm file."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.hydroas.file_io.gmi_parser import CARD_INDEX, FIRST_FIELD

ENTITY_NAMES = ['points', 'arcs', 'polygons']


def fix(cards: list):
    """
    Fix various issues.

    The .2dm format has some "interesting" evolutions in it that make working with it complicated.
    This function cleans some of them up so the builder can pretend that things are more uniform.

    Args:
        cards: Cards parsed from the .2dm.
    """
    _rewrite_entity_ids(cards)
    _rewrite_value(cards)
    _concatenate_nodestrings(cards)
    _name_nodestrings(cards)


def _rewrite_entity_ids(cards: list):
    """
    Rewrite entity IDs so they're all consistent.

    Cards containing entity IDs express them in various inconsistent ways, and they're all different from how DMI
    represents them. This rewrites them all to be consistent and match the DMI.
    """
    entities = {
        0: 'points',
        'N': 'points',
        1: 'arcs',
        'S': 'arcs',
        2: 'polygons',
        'E': 'polygons',
    }

    for card in cards:
        if not card:
            continue
        if card[CARD_INDEX] in ['bc', 'bc_val', 'befont']:
            card[1] = entities[card[1]] if card[1] in entities else None
        elif card[CARD_INDEX] == 'disp_opts':
            card[2] = entities[card[2]] if card[2] in entities else None
        elif card[CARD_INDEX] == 'bc_disp_opts':
            card[2] = entities[card[2]] if card[2] in entities else None


def _rewrite_value(cards: list[list]):
    """
    Rewrite VALUE to FLOAT.

    SMS used to write VALUE instead of FLOAT for some cards. We accept VALUE for compatibility, but don't
    want to deal with it later, so this gets rid of it now.

    Args:
        cards: Cards parsed from the .2dm.
    """
    cards_to_fix = [
        'bc_def',
        'gp_def',
        'mat_def',
        'bc_val',
        'gp_val',
        'mat_val',
    ]
    cards_to_indexes = {
        'bc_def': 8,
        'gp_def': 8,
        'mat_def': 8,
        'bc_val': 5,
        'gp_val': 3,
        'mat_val': 4,
    }
    for card in cards:
        if card and card[CARD_INDEX] in cards_to_fix:
            i = cards_to_indexes[card[CARD_INDEX]]
            if i < len(card) and card[i] == 'VALUE':
                card[i] = 'FLOAT'


def _name_nodestrings(cards: list):
    """
    Convert unnamed nodestrings into named ones.

    NS cards have an optional name on the end. This inserts omitted ones so following
    code can assume nodestrings are always named.

    Args:
        cards: Cards parsed from the .2dm.
    """
    for card in cards:
        if card and card[CARD_INDEX] == 'ns' and isinstance(card[-1], int):
            card.append('')


def _concatenate_nodestrings(cards: list[list]):
    """
    Convert multi-line nodestrings into single-line ones.

    There's a comment in the wiki for .2dm files that "A nodestring may consist of more than ten nodes which constitute
    a file line, consequently a nodestring may extend multiple NS cards." I have no idea why there was a ten field
    limit, but there's really no reason for it at this point, so this just concatenates them to save the builder the
    trouble of dealing with that oddity.

    Args:
        cards: Cards parsed from the .2dm.
    """
    for i in range(len(cards)):
        if not cards[i] or cards[i][CARD_INDEX] != 'ns':
            continue

        j = i + 1
        while isinstance(cards[i][-1], int) and cards[i][-2] > 0 and j < len(cards):
            cards[i].extend(cards[j][FIRST_FIELD:])
            cards[j] = []
            j += 1
