"""Utilitiy functions to deal with query."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
import uuid

# 2. Third party modules
import orjson

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import UGridItem
from xms.api.dmi import ActionRequest, MenuItem, Query
from xms.api.tree import tree_util
from xms.api.tree.tree_node import TreeNode
from xms.components.display.display_options_io import read_display_options_from_json, write_display_options_to_json
from xms.core.filesystem import filesystem as fs
from xms.data_objects.parameters import Component
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.testing import tools

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.file_io import io_util


def package_mainfile_from_query(query, unique_name):
    """Returns the package main file.

    Assumes query context is at the hidden sim component.

    Args:
        query: a Query object to communicate with GMS.
        unique_name (str): The unique_name from the xml file (e.g. 'TDIS', 'IMS')

    Returns:
        See description.
    """
    sim_uuid = query.current_item_uuid()
    sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, sim_uuid)
    return package_mainfile_from_sim_node(sim_node, unique_name)


def package_mainfile_from_sim_node(sim_node: TreeNode, unique_name: str) -> str:
    """Returns the package main file.

    Assumes query context is at the hidden sim component.

    Args:
        sim_node: The simulation tree node.
        unique_name: The unique_name from the xml file (e.g. 'TDIS', 'IMS')

    Returns:
        See description.
    """
    item = tree_util.descendants_of_type(
        sim_node, xms_types=['TI_COMPONENT'], unique_name=unique_name, model_name='MODFLOW 6', only_first=True
    )
    if item:
        return item.main_file
    return ''


def ensure_disp_opts_file_exists(main_file, json_file):
    """Copy the default display options json file if needed.

    Args:
        main_file (str): File location of the component's main file.
        json_file (str): Filepath to json file.
    """
    my_disp_opts = os.path.join(os.path.dirname(main_file), json_file)
    if not os.path.isfile(my_disp_opts):
        default_disp_opts = os.path.join(os.path.dirname(__file__), 'resources', 'display_options', json_file)
        json_dict = read_display_options_from_json(default_disp_opts)  # Read the default options
        categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
        categories.from_dict(json_dict)
        categories.comp_uuid = io_util.uuid_from_path(my_disp_opts)
        write_display_options_to_json(my_disp_opts, categories)


def get_display_option_filenames(main_file: str | Path):
    """Returns a list of the display options files given the component's main file.

    Args:
        main_file: main file path.

    Returns:
        See description.
    """
    display_options_files = []
    dir_ = os.path.dirname(main_file)
    for _, _, files in os.walk(dir_):
        for f in files:
            if f.endswith('_display_options.json'):
                display_options_files.append(os.path.join(dir_, f))
    return display_options_files


def ugrid_uuid_from_model_node(model_node: TreeNode) -> str:
    """Get the UGrid TreeNode from parent GWF or GWT TreeNode.

    Args:
        model_node: The GWF or GWT project explorer tree node

    Returns:
        str: UUID of the linked UGrid, empty string if not found
    """
    if not model_node:
        return ''
    for child in model_node.children:
        if child.is_ptr_item and type(child.data) is UGridItem:
            return child.uuid
    return ''


def get_ugrid_filename_and_units(ugrid_uuid: str, query: Query):
    """Returns the filename where the UGrid is saved.

    Args:
        ugrid_uuid: uuid of the UGrid.
        query : Object for communicating with GMS

    Returns:
        See description.
    """
    dogrid = query.item_with_uuid(ugrid_uuid)  # Get the data_objects UGrid
    filename = ''
    hunits = ''
    if dogrid:
        filename = dogrid.cogrid_file
        proj = dogrid.projection  # This is always the display projection
        hunits = proj.horizontal_units
    return filename, hunits


def save_to_location_common(main_file, new_path, save_type):
    """Duplicate code in text_editor_component and cbc_component is now here.

    Args:
        main_file (str): Path to component's main file.
        new_path (str): Path to the new save location.
        save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
            DUPLICATE - happens when the tree item owner is duplicated. The new component will always be unlocked to
            start with.

            PACKAGE - happens when the project is being saved as a package. As such, all data must be copied and all
            data must use relative file paths.

            SAVE - happens when re-saving this project.

            SAVE_AS - happens when saving a project in a new location. This happens the first time we save a
            project.

            UNLOCK - happens when the component is about to be changed and it does not have a matching uuid folder
            in the temp area. May happen on project read if the XML specifies to unlock by default.

    Returns:
        (tuple): tuple containing:
            - messages (list of tuple of str): List of tuples with the first element of the
              tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
              text.
            - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
    """
    # Check if we are already in the new location
    new_main_file = os.path.normpath(os.path.join(new_path, os.path.basename(main_file)))
    if fs.paths_are_equal(new_main_file, main_file):
        return main_file, [], []

    # Fix the paths and rewrite the file
    rewrite = False
    with open(new_main_file, 'rb') as file:
        cards = orjson.loads(file.read())
        relative_path = cards.get('SOLUTION_FILE_RELATIVE', '')
        if relative_path:
            rewrite = True
            new_full_path = fs.resolve_relative_path(os.path.dirname(main_file), relative_path)
            cards['SOLUTION_FILE_FULL'] = new_full_path
            cards['SOLUTION_FILE_RELATIVE'] = fs.compute_relative_path(new_path, new_full_path)

    if rewrite:
        with open(new_main_file, 'wb') as file:
            data = orjson.dumps(cards)
            file.write(data)

    return new_main_file, [], []


def find_solution_file(solution_file, solution_file_full, solution_file_relative, main_file):
    """Returns the location of the solution file or '' if not found.

    We store both the full path and the relative path from the component uuid dir because there are multiple
    scenarios:
    1. Entire project might be moved to a new computer/location. Full path won't work, only relative will
    2. In save_to_location, we need the full path to compute the new relative path

    Args:
        solution_file (str): Old thing we used to store.
        solution_file_full (str): Full path to the solution file.
        solution_file_relative (str): Relative path to the solution file from main_file directory.
        main_file (str): Main file of component.

    Returns:
        (str): See description.
    """
    if solution_file:  # This was the old way
        full_path = fs.resolve_relative_path(os.path.dirname(main_file), solution_file)
        if full_path and os.path.isfile(full_path):
            return full_path
    if solution_file_full and os.path.isfile(solution_file_full):
        return solution_file_full
    full_path = fs.resolve_relative_path(os.path.dirname(main_file), solution_file_relative)
    if full_path and os.path.isfile(full_path):
        return full_path
    return ''


def build_solution_component(text_file, sim_uuid, model_uuid, component_dir, unique_name) -> Component:
    """Create a component for solution files.

    Args:
        text_file (str): Filesystem path to the solution file.
        sim_uuid (str): The UUID of the simulation that this solution belongs to.
        model_uuid (str): The UUID of the model that this solution component belongs to.
        component_dir (str): The component directory to add the main file to.
        unique_name (str): XML definition unique_name of the component to build

    Returns:
        The data_objects component to send back to XMS for the new component.
    """
    comp_uuid = tools.new_uuid()
    base_filename = os.path.basename(text_file)
    # create the new main file
    uuid_path = os.path.join(component_dir, comp_uuid)
    os.makedirs(uuid_path)
    main_filepath = os.path.join(uuid_path, f'solution.{unique_name}')
    solution_file = text_file.strip()
    relative_path = fs.compute_relative_path(uuid_path, solution_file)
    # We store both the full path and the relative path from the component uuid dir because there are multiple
    # scenarios:
    # 1. Entire project might be moved to a new computer/location. Full path won't work, only relative will
    # 2. In save_to_location, we need the full path to compute the new relative path
    cards = {
        'SIM_UUID': sim_uuid,
        'MODEL_UUID': model_uuid,
        'SOLUTION_FILE_FULL': solution_file,
        'SOLUTION_FILE_RELATIVE': relative_path
    }
    with open(main_filepath, 'wb') as file:
        data = orjson.dumps(cards)
        file.write(data)
    do_comp = Component(
        name=base_filename,
        comp_uuid=comp_uuid,
        main_file=main_filepath,
        model_name='MODFLOW 6',
        unique_name=unique_name,
        locked=True
    )
    return do_comp


def add_tree_menu_command(menu_text, method_name, main_file, class_name, module_name, sim_uuid, menu_list, icon=''):
    """Adds a command to the menu.

    Args:
        menu_text (str): Text of the menu command.
        method_name (str): Name of method to call on the class (class_name).
        main_file (str): File path of component main file.
        class_name (str): Name of component class.
        module_name (str): Python module name.
        sim_uuid (str): UUID of the simulation for solution load.
        menu_list (list of xmsapi.dmi.MenuItem): List of menu commands.
        icon (str): Name of the icon, like 'save.svg'.
    """
    # Create the ActionRequest
    action = ActionRequest(
        main_file=main_file,
        modality='modal',
        class_name=class_name,
        module_name=module_name,
        method_name=method_name,
        sim_uuid=sim_uuid,
        parameters={'main_file': main_file}
    )

    # Create the MenuItem
    icon_path = icon if icon else None
    menu_item = MenuItem(text=menu_text, action=action, icon_path=icon_path)
    menu_list.append(menu_item)


def add_selected_cells_menu_command(
    menu_text, method_name, main_file, class_name, module_name, menu_list, icon='', selection=None
):
    """Adds a command to the menu.

    Args:
        menu_text (str): Text of the menu command.
        method_name (str): Name of method to call on the class (class_name).
        main_file (str): File path of component main file.
        class_name (str): Name of component class.
        module_name (str): Python module name.
        menu_list (list of xmsapi.dmi.MenuItem): List of menu commands.
        icon (str): Name of the icon, like 'save.svg'.
        selection (list[str]): selection.
    """
    # Create the ActionRequest
    action = ActionRequest(
        main_file=main_file,
        modality='modal',
        class_name=class_name,
        module_name=module_name,
        method_name=method_name,
        parameters={'selection': selection['CELL']}
    )

    # Create the MenuItem
    icon_path = icon if icon else None
    menu_item = MenuItem(text=menu_text, action=action, icon_path=icon_path)
    menu_list.append(menu_item)


def get_selected_cells(ugrid_uuid, query):
    """Returns the selected cells."""
    selection = query.get_ugrid_selection(ugrid_uuid)
    return selection.get('CELL', [])


def get_dataset_active_timestep_time(dset_uuid, query):
    """Returns the active timestep index of the active dataset."""
    dset = query.item_with_uuid(dset_uuid)
    dset_node = tree_util.find_tree_node_by_uuid(query.project_tree, dset_uuid)
    if not dset or not dset_node:
        return 0.0
    return dset.times[dset_node.active_timestep]


def duplicate_display_opts(new_main_file: str | Path) -> None:
    """Duplicates display options, sets the component UUID and generates a new display list UUID.

    Args:
        new_main_file (str | Path): Location of the new component. Assumes the display options files have already been
            copied to the new locaiton.
    """
    new_path = str(Path(new_main_file).parent)
    for disp_file in Path(new_main_file).parent.rglob('*_display_options.json'):
        json_dict = read_display_options_from_json(disp_file)
        if 'uuid' in json_dict:
            json_dict['uuid'] = str(uuid.uuid4())
            json_dict['comp_uuid'] = os.path.basename(new_path)
            categories = CategoryDisplayOptionList()
            categories.from_dict(json_dict)
            write_display_options_to_json(disp_file, categories)


def update_display_action(model_ftype: str, model_mainfile: str | Path, comp_main_file: str | Path) -> ActionRequest:
    """Creates and returns an ActionRequest to update the display."""
    # Create an ActionRequest to trigger update of display list. We need to send back an ActionRequest
    # because we do not know the UGrid's UUID.
    component_uuid = io_util.uuid_from_path(model_mainfile)
    # Add all display options files
    filenames = get_display_option_filenames(comp_main_file)
    py_defs = data_util.model_class_module()
    action = ActionRequest(
        main_file=str(model_mainfile),
        modality='no_dialog',
        class_name=py_defs[model_ftype][0],
        module_name=py_defs[model_ftype][1],
        method_name='get_initial_display_options',
        comp_uuid=component_uuid,
        parameters={'filenames': filenames}
    )
    return action


def is_locked(node: TreeNode, query: Query) -> bool:
    """Return True if component is locked, else False.

    Args:
        node: A tree node.
        query: XMS Query.

    Returns:
        See description.
    """
    comp = query.item_with_uuid(node.uuid)
    return comp.locked if comp else True  # Usages just continue in loops if getting component failed.
