"""GncData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util
from xms.mf6.gui.options_defs import Checkbox


class GncData(ListPackageData):
    """Data class to hold the info from a list package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'GNC6'
        self.numalphaj = 0  # NUMALPHAJ

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the DIS package in use and the AUX variables.
        The package specific and AUX columns are type object because they
        might contain time series strings.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        if block.upper() == 'GNCDATA':
            columns = {
                'CELLIDN': (np.int32, 1),
                'CELLIDM': (np.int32, 1),
                'CELLIDSJ': (np.int32, 1),
                'ALPHASJ': (np.float64, 0.0),
            }
        else:
            raise AssertionError()  # Error!

        names, types, defaults = gui_util.column_info_tuple_from_dict(columns)
        return names, types, defaults

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        if block.upper() == 'GNCDATA':
            names, _types, _defaults = self.get_column_info(block)
            return {
                names.index('CELLIDN'): 'Cellid of the cell, n, in which the ghost node is located',
                names.index('CELLIDM'): 'Cellid of the connecting cell, m, to which flow occurs from the ghost'
                                        ' node',
                names.index('CELLIDSJ'):
                    'Array of CELLIDS for the contributing j cells, which contribute to the'
                    ' interpolated head value at the ghost node',
                names.index('ALPHASJ'): 'Contributing factors for each contributing node in CELLIDSJ',
            }

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        return [], {}, {}

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Ghost Node Correction (GNC) Package'

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                Checkbox('PRINT_INPUT', 'Print input to listing file'),
                Checkbox('PRINT_FLOWS', 'Print flows to listing file'),
                Checkbox('EXPLICIT', 'Apply ghost node correction explicitly'),
            ]
        )
