"""MvrData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxField


class MvrData(ListPackageData):
    """Data class to hold the info from a MVR package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'MVR6'
        self.maxpackages = 0  # MAXPACKAGES
        self.block_with_cellids = ''
        self.list_blocks = {'PACKAGES': ''}

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the DIS package in use and the AUX variables.
        The package specific and AUX columns are type object because they
        might contain time series strings.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        modelnames = self.options_block.has('MODELNAMES')
        if block.upper() == 'PACKAGES':
            if modelnames:
                columns = {
                    'MNAME': (object, ''),
                    'PNAME': (object, ''),
                }
            else:
                columns = {
                    'PNAME': (object, ''),
                }
            names, types, defaults = gui_util.column_info_tuple_from_dict(columns)
            return names, types, defaults
        else:
            return self.package_column_info()

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        # yapf: disable
        if block.upper() == 'PACKAGES':
            tool_tips = {}
            if 'MNAME' in names:
                tool_tips[names.index('MNAME')] = 'Name of model containing the package'
            if 'PNAME' in names:
                tool_tips[
                    names.index('PNAME')
                ] = 'Name of a package that may be included in a subsequent stress period block'
            return tool_tips
        else:  # stress periods
            tool_tips = {}
            if 'MNAME1' in names:
                tool_tips[names.index('MNAME1')] = 'Name of model containing the package, PNAME1'
            if 'PNAME1' in names:
                tool_tips[names.index('PNAME1')] = 'Package name for the provider'
            if 'ID1' in names:
                tool_tips[names.index('ID1')] = 'Identifier for the provider'
            if 'MNAME2' in names:
                tool_tips[names.index('MNAME2')] = 'Name of model containing the package, PNAME2'
            if 'PNAME2' in names:
                tool_tips[names.index('PNAME2')] = 'Package name for the receiver.'
            if 'ID2' in names:
                tool_tips[names.index('ID2')] = 'Identifier for the receiver'
            if 'MVRTYPE' in names:
                tool_tips[
                    names.index('MVRTYPE')
                ] = 'Character string signifying the method for determining how much water will be moved'
            if 'VALUE' in names:
                tool_tips[
                    names.index('VALUE')
                ] = 'Value to be used in the equation for calculating the amount of water to move'
            # yapf: enable
            return tool_tips

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        modelnames = self.options_block.has('MODELNAMES')
        if modelnames:
            columns = {
                'MNAME1': (object, ''),
                'PNAME1': (object, ''),
                'ID1': (np.int32, 1),
                'MNAME2': (object, ''),
                'PNAME2': (object, ''),
                'ID2': (np.int32, 1),
                'MVRTYPE': (object, ''),
                'VALUE': (np.float64, 0.0),
            }
        else:
            columns = {
                'PNAME1': (object, ''),
                'ID1': (np.int32, 1),
                'PNAME2': (object, ''),
                'ID2': (np.int32, 1),
                'MVRTYPE': (object, ''),
                'VALUE': (np.float64, 0.0),
            }
        names, types, defaults = gui_util.column_info_tuple_from_dict(columns)

        return names, types, defaults

    # @overrides
    def get_column_delegate_info(self, block):
        """Returns a list of tuples of [0] column index and [1] list of strings."""
        delegate_info = None
        if block == 'PERIODS':
            delegate_info = [(3, ['FRACTION', 'EXCESS', 'THRESHOLD', 'UPTO'])]

        return delegate_info

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Water Mover (MVR) Package'

    @override
    def get_time_series_columns(self) -> list[int]:
        """Returns a list of the column indices that can contain time series.

        Returns:
            List of indices of columns that can contain time series.
        """
        return []

    def stress_id_columns(self):
        """Returns the column name where the id exists that can be used to help identify this stress across periods.

        Typically is 'CELLIDX' which is added by GMS but is 'RNO' for SFR.

        Returns:
            See description.
        """
        return ['ID1', 'ID2']

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        return OptionsBlock(
            [
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
                Checkbox('PRINT_FLOWS', brief='Print flows to listing file'),
                Checkbox(
                    'MODELNAMES',
                    brief='All package names will be preceded by the model name for the package',
                    check_box_method='on_chk_modelnames'
                ),
                CheckboxField('BUDGET FILEOUT', brief='Save budget to file', type_='str'),
                CheckboxField('BUDGETCSV FILEOUT', brief='Save budget to CSV file', type_='str'),
            ]
        )
