"""ImsData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.data.solution_data import SolutionData
from xms.mf6.gui.options_defs import CheckboxComboBox, CheckboxField


class ImsData(SolutionData):
    """Class to hold the IMS6 package data."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'IMS6'
        self.slnmnames = []  # model names, in mfsim.nam, which are also mname

        # Default the variables to the 'SIMPLE' option

        # Nonlinear
        self.outer_dvclose_use = False
        self.outer_dvclose = 0.001
        self.outer_maximum_use = False
        self.outer_maximum = 25
        self.under_relaxation_use = False
        self.under_relaxation = 'NONE'  # ['NONE', 'SIMPLE', 'COOLEY', 'DBD']
        self.under_relaxation_theta_use = False
        self.under_relaxation_theta = 1.0
        self.under_relaxation_kappa_use = False
        self.under_relaxation_kappa = 0.0
        self.under_relaxation_gamma_use = False
        self.under_relaxation_gamma = 0.0
        self.under_relaxation_momentum_use = False
        self.under_relaxation_momentum = 0.0
        self.backtracking_number_use = False
        self.backtracking_number = 0
        self.backtracking_tolerance_use = False
        self.backtracking_tolerance = 0.0
        self.backtracking_reduction_factor_use = False
        self.backtracking_reduction_factor = 0.0
        self.backtracking_residual_limit_use = False
        self.backtracking_residual_limit = 0.0
        self.linear_solver_use = False
        self.linear_solver = 'SAMG'  # SAMG extension to standard MODFLOW 6

        # Linear
        self.inner_maximum_use = False
        self.inner_maximum = 50
        self.inner_dvclose_use = False
        self.inner_dvclose = 0.001
        self.inner_rclose_use = False
        self.inner_rclose = 0.1
        self.rclose_option_use = False
        self.rclose_option = 'STRICT'  # ['STRICT', 'L2NORM_RCLOSE', 'RELATIVE_RCLOSE']
        self.linear_acceleration_use = False
        self.linear_acceleration = 'CG'  # ['CG', 'BICGSTAB']
        self.relaxation_factor_use = False
        self.relaxation_factor = 0.0
        self.preconditioner_levels_use = False
        self.preconditioner_levels = 0
        self.preconditioner_drop_tolerance_use = False
        self.preconditioner_drop_tolerance = 0.0
        self.number_orthogonalizations_use = False
        self.number_orthogonalizations = 0
        self.scaling_method_use = False
        self.scaling_method = 'NONE'  # ['NONE', 'DIAGONAL', 'L2NORM']
        self.reordering_method_use = False
        self.reordering_method = 'NONE'  # ['NONE', 'RCM', 'M

    def get_complexity_variable_defaults(self):
        """Returns a dict containing the default values for each complexity (SIMPLE, MODERATE, COMPLEX).

        Returns:
            See description.
        """
        #           NONLINEAR     [SIMPLE, MODERATE, COMPLEX]
        preset = {
            'outer_dvclose': [0.001, 0.01, 0.1],
            'outer_maximum': [25, 50, 100],
            'under_relaxation': ['NONE', 'DBD', 'DBD'],
            'under_relaxation_theta': [0.0, 0.9, 0.8],
            'under_relaxation_kappa': [0.0, 0.0001, 0.0001],
            'under_relaxation_gamma': [0.0, 0.0, 0.0],
            'under_relaxation_momentum': [0.0, 0.0, 0.0],
            'backtracking_number': [0, 0, 20],
            'backtracking_tolerance': [0.0, 0.0, 1.05],
            'backtracking_reduction_factor': [0.0, 0.0, 0.1],
            'backtracking_residual_limit': [0.0, 0.0, 0.002],
            # LINEAR
            'inner_maximum': [50, 100, 500],
            'inner_dvclose': [0.001, 0.01, 0.1],
            'inner_rclose': [0.1, 0.1, 0.1],
            'rclose_option': ['STRICT', 'STRICT', 'STRICT'],
            'linear_acceleration': ['CG', 'BICGSTAB', 'BICGSTAB'],
            'relaxation_factor': [0.0, 0.97, 0.0],
            'preconditioner_levels': [0, 0, 5],
            'preconditioner_drop_tolerance': [0.0, 0.0, 0.0001],
            'number_orthogonalizations': [0, 0, 2],
            'scaling_method': ['NONE', 'NONE', 'NONE'],
            'reordering_method': ['NONE', 'NONE', 'NONE'],
        }
        return preset

    def set_complexity(self, index):
        """Updates class member values based on the complexity setting SIMPLE, MODERATE, COMPLEX, or NOT SPECIFIED.

        Also turns off all 'Use' variables.

        Args:
            index (int): 0 - SIMPLE, 1 - MODERATE, 2 - COMPLEX.
        """
        my_set = {0, 1, 2}
        if index not in my_set:
            return

        preset = self.get_complexity_variable_defaults()
        for key, val in preset.items():
            setattr(self, f'{key}_use', False)
            setattr(self, key, val[index])

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Iterative Model Solution (IMS)'

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        ats_brief = 'Fraction of the maximum allowable outer iterations used with ATS utility if it is active'
        return OptionsBlock(
            [
                CheckboxComboBox(
                    'PRINT_OPTION',
                    brief='Print convergence information',
                    items=['NONE', 'SUMMARY', 'ALL'],
                    value='NONE'
                ),
                CheckboxComboBox(
                    'COMPLEXITY',
                    brief='Default solver parameters',
                    items=['SIMPLE', 'MODERATE', 'COMPLEX'],
                    value='SIMPLE'
                ),
                CheckboxField('CSV_OUTER_OUTPUT FILEOUT', brief='Save outer iteration convergence info', type_='str'),
                CheckboxField('CSV_INNER_OUTPUT FILEOUT', brief='Save inner iteration convergence info', type_='str'),
                CheckboxComboBox(
                    'NO_PTC', brief='Disable pseudo-transient continuation', items=['ALL', 'FIRST'], value='ALL'
                ),
                CheckboxField('ATS_OUTER_MAXIMUM_FRACTION', brief=ats_brief, type_='float', value=0.0),
            ]
        )
