"""GwfData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import Grid
from xms.core.filesystem import filesystem as fs
from xms.data_objects.parameters import UGrid as DoGrid

# 4. Local modules
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.data.data_type_aliases import DisX
from xms.mf6.data.grid_info import GridInfo
from xms.mf6.file_io import io_util


class ModelDataBase(BaseFileData):
    """A GWF/GWT/GWE model name file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            mtype (str): Type of model (GWF6) as used in the simulation name file.
            mname (str): User-assigned name of the model as used in the simulation name file.
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = kwargs.get('ftype', '')
        self.model = self
        self.mtype = kwargs.get('mtype', self.ftype)
        self.mname = kwargs.get('mname', '')
        self.packages = []  # List of packages
        self.gui_edit_active = False  # being edited by gui
        self._projection_wkt = ''  # Projection well known text
        self._uuid = ''
        self.can_import = False
        self.can_export = False

    def get_dogrid(self) -> DoGrid | None:
        """Returns the data_objects grid associated with the model.

        Returns:
            See description.
        """
        if self.mfsim and self.mfsim.xms_data:
            return self.mfsim.xms_data.get_dogrid(self._get_uuid())
        return None

    def set_dogrid(self, dogrid: DoGrid) -> None:
        """Sets the data_objects grid associated with the model.

        Args:
            dogrid: The data_objects grid.
        """
        if self.mfsim and self.mfsim.xms_data:
            self.mfsim.xms_data.set_dogrid(self._get_uuid(), dogrid)

    def get_cogrid(self) -> Grid | None:
        """Return the constrained grid associated with the model.

        Returns:
            See description.
        """
        return self.mfsim.xms_data.get_cogrid(self._get_uuid())

    def set_cogrid(self, cogrid: Grid) -> None:
        """Sets the constrained grid associated with the model.

        Args:
            cogrid: The constrained grid.
        """
        self.mfsim.xms_data.set_cogrid(self._get_uuid(), cogrid)

    @property
    def projection_wkt(self) -> str:
        """Return the projection well known text associated with the dis* package / grid."""
        return self._projection_wkt

    @projection_wkt.setter
    def projection_wkt(self, wkt: str) -> None:
        """Sets the projection well known text associated with the dis* package / grid."""
        self._projection_wkt = wkt

    def _get_uuid(self) -> str:
        """Returns this packages uuid, if possible, else ''.

        Returns:
            See description.
        """
        if not self._uuid:
            if self.tree_node:
                self._uuid = self.tree_node.uuid
            else:
                self._uuid = io_util.uuid_from_path(self.filename)
        return self._uuid

    def add_package(self, package_data):
        """Appends the package to the end of the list.

        Args:
            package_data: Something derived from BaseFileData
        """
        self.packages.append(package_data)

    def grid_info(self) -> GridInfo | None:
        """Returns a reference to the GridInfo saved with the DIS (DISV, DISU) package.

        Returns:
            (GridInfo): See description.
        """
        if not self.packages:
            raise RuntimeError('Error in ModelDataBase.grid_info(). No packages.')  # pragma no cover - shouldn't happen

        if self.packages and self.packages[0].ftype in {'DIS6', 'DISV6', 'DISU6'}:
            return self.packages[0].grid_info()  # DIS/DISV/DISU package should always be first
        return None

    def get_dis(self) -> DisX | None:
        """Returns the DIS (DISV, DISU) package for this mode.

        Returns:
            See description.
        """
        if not self.packages:
            raise RuntimeError('Error in GwfData.get_dis(). No packages.')  # pragma no cover - shouldn't happen

        if self.packages and self.packages[0].ftype in {'DIS6', 'DISV6', 'DISU6'}:
            return self.packages[0]  # DIS/DISV/DISU package should always be first
        return None

    def package_from_filename(self, filename, testing=False):
        """Returns the package with the given filename, or None if not found.

        Args:
            filename (str): Path of the package file.

        Returns:
            The package data object.
        """
        for package in self.packages:
            package_filename = os.path.basename(package.filename) if testing else package.filename
            if fs.paths_are_equal(package_filename, filename):
                return package
        return None

    def packages_from_ftype(self, ftype):
        """Returns the packages with the given ftype.

        Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')

        Returns:
            The list of packages.
        """
        packages = []
        for package in self.packages:
            if package.ftype == ftype:
                packages.append(package)
        return packages
