"""ArrayPackageReader class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.array import Array
from xms.mf6.data.array_layer import ArrayLayer
from xms.mf6.file_io import io_util
from xms.mf6.file_io.array_layer_reader import ArrayLayerReader
from xms.mf6.file_io.package_reader_base import PackageReaderBase


class ArrayPackageReader(PackageReaderBase):
    """Reads an array-based package file (RCH, EVT)."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self._aux: list[str] | None = None  # Aux variables
        self._tab_names = []  # Will become ArrayPackageData._griddata_names + self._aux_vars

    def _on_end_block(self, block_name):
        """Save arrays to external files when we finish reading a stress period.

        Args:
            block_name (str): block name from file
        """
        if block_name != 'PERIOD':
            return

        sp = self._curr_block_number
        if sp not in self._data.period_data:
            self._data.add_period(sp)
            return

        if not PackageReaderBase.importing:
            return

        flist = PackageReaderBase.temp_file_list
        self._data.period_data[sp].dump_to_temp_files(flist)

    def _read_period(self, line):  # noqa: C901  (too complex)
        """Reads the stress period, storing the external file path.

        Args:
            line: A line from the file.
        """
        if not self._tab_names:
            self._tab_names = self._data.tab_names()
        if self._aux is None:
            self._aux = self._data.options_block.get('AUXILIARY', [])

        sp = self._curr_block_number
        read = False
        self._data.add_period(sp)
        while True:
            if read:
                line = self._file.readline()
                if not line:
                    return False  # Unexpected eof
            if io_util.is_end_line(line):
                break
            else:
                words = line.split()
                if words and words[0].upper() in [x.upper() for x in self._tab_names]:
                    array_name = words[0].upper()

                    # If it's an aux variable, make it the same case as in the AUXILIARY option
                    if self._aux:
                        for item in self._aux:
                            if words[0].upper() == item.upper():
                                array_name = item

                    array = Array(array_name=array_name, layered=True)

                    if len(words) > 2 and words[1].upper() == 'TIMEARRAYSERIES':
                        array.storage = 'TIME-ARRAY SERIES'
                        array.time_array_series = words[2]
                        read = True
                    else:
                        array_layer_reader = ArrayLayerReader()
                        grid_info = copy.deepcopy(self._data.grid_info())
                        grid_info.nlay = 1
                        nvalues, shape = ArrayLayer.number_of_values_and_shape(True, grid_info)
                        array_layer, line = array_layer_reader.read(
                            self._data.filename, nvalues, self._file, PackageReaderBase.importing
                        )
                        array.storage = array_layer.storage
                        if self._data.is_layer_indicator(array_name):
                            array_layer.numeric_type = 'int'
                        array_layer.name = array.array_name
                        array_layer.shape = shape
                        array.layers.append(array_layer)
                        read = (len(line) == 0)

                    self._data.period_data[sp].add_array(array)

        # Stop reading if this is the last stress period
        if sp == self._data.nper():
            self._stop_after_block = 'PERIOD'

        return True
