"""Functions to get file readers and writers."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


def reader_exists(ftype):
    """Returns True if a reader exists for the given ftype.

    Args:
        ftype (str): The file type used in the model name file (e.g. 'WEL6')

    Returns:
        (bool): See description.
    """
    reader = get_reader(ftype)
    return reader is not None


# @staticmethod  # noqa: C901  (too complex)
def get_reader(ftype):
    """Gets the package reader from the ftype and sets the grid_dimension and nper variables.

    Imports are embedded in the if-else block to avoid circular
    dependencies encountered when listing them all at the top of the file.

    Args:
        ftype (str): The file type used in the model name file (e.g. 'WEL6')

    Returns:
        reader: objects that will read a modflow file
    """
    # Simulation
    if ftype == 'MFSIM6':
        from xms.mf6.file_io.mfsim_reader import MfsimReader
        return MfsimReader()

    # Simulation children
    if ftype == 'TDIS6':
        from xms.mf6.file_io.tdis_reader import TdisReader
        return TdisReader()
    elif ftype == 'IMS6':
        from xms.mf6.file_io.gwf.ims_reader import ImsReader
        return ImsReader()
    elif ftype == 'EMS6':
        from xms.mf6.file_io.gwf.ems_reader import EmsReader
        return EmsReader()

    # Exchanges
    elif ftype in {'GWF6-GWF6', 'GWT6-GWT6', 'GWE6-GWE6'}:
        from xms.mf6.file_io.exchanges.gwx_gwx_reader import GwxGwxReader
        return GwxGwxReader(ftype=ftype)
    elif ftype in {'GWF6-GWT6', 'GWF6-GWE6', 'GWF6-PRT6'}:
        from xms.mf6.file_io.exchanges.exchange_reader_base import ExchangeReaderBase
        return ExchangeReaderBase(ftype=ftype)

    # GWF
    elif ftype == 'ATS6':
        from xms.mf6.file_io.ats_reader import AtsReader
        return AtsReader()
    elif ftype == 'BUY6':
        from xms.mf6.file_io.gwf.buy_reader import BuyReader
        return BuyReader()
    elif ftype == 'CHD6':
        from xms.mf6.file_io.package_reader import PackageReader
        return PackageReader(ftype=ftype)
    elif ftype == 'CSUB6':
        from xms.mf6.file_io.gwf.csub_reader import CsubReader
        return CsubReader()
    elif ftype == 'DIS6':
        from xms.mf6.file_io.dis_reader import DisReader
        return DisReader()
    elif ftype == 'DISU6':
        from xms.mf6.file_io.disu_reader import DisuReader
        return DisuReader()
    elif ftype == 'DISV6':
        from xms.mf6.file_io.disv_reader import DisvReader
        return DisvReader()
    elif ftype == 'DRN6':
        from xms.mf6.file_io.package_reader import PackageReader
        return PackageReader(ftype=ftype)
    elif ftype == 'EVT6':
        from xms.mf6.file_io.gwf.evt_reader import EvtReader
        return EvtReader()
    elif ftype == 'EVTA6':
        from xms.mf6.file_io.gwf.evt_reader import EvtReader
        return EvtReader()
    elif ftype == 'GHB6':
        from xms.mf6.file_io.package_reader import PackageReader
        return PackageReader(ftype=ftype)
    elif ftype == 'GNC6':
        from xms.mf6.file_io.gwf.gnc_reader import GncReader
        return GncReader()
    elif ftype == 'GWF6':
        from xms.mf6.file_io.gwf.gwf_reader import GwfReader
        return GwfReader()
    elif ftype == 'HFB6':
        from xms.mf6.file_io.package_reader import PackageReader
        return PackageReader(ftype=ftype)
    elif ftype == 'IC6':
        from xms.mf6.file_io.ic_reader import IcReader
        return IcReader()
    elif ftype == 'LAK6':
        from xms.mf6.file_io.gwf.lak_reader import LakReader
        return LakReader()
    elif ftype == 'MAW6':
        from xms.mf6.file_io.gwf.maw_reader import MawReader
        return MawReader()
    elif ftype == 'MVR6':
        from xms.mf6.file_io.gwf.mvr_reader import MvrReader
        return MvrReader()
    elif ftype == 'NPF6':
        from xms.mf6.file_io.gwf.npf_reader import NpfReader
        return NpfReader()
    elif ftype == 'OBS6':
        from xms.mf6.file_io.obs_reader import ObsReader
        return ObsReader()
    elif ftype == 'OC6':
        from xms.mf6.file_io.gwf.oc_reader import OcReader
        return OcReader()
    elif ftype == 'POBS6':
        from xms.mf6.file_io.pest.pest_obs_reader import PestObsReader
        return PestObsReader()
    elif ftype == 'RCH6':
        from xms.mf6.file_io.gwf.rch_reader import RchReader
        return RchReader()
    elif ftype == 'RCHA6':
        from xms.mf6.file_io.gwf.rch_reader import RchReader
        return RchReader()
    elif ftype == 'RIV6':
        from xms.mf6.file_io.package_reader import PackageReader
        return PackageReader(ftype=ftype)
    elif ftype == 'SFR6':
        from xms.mf6.file_io.gwf.sfr_reader import SfrReader
        return SfrReader()
    elif ftype == 'STO6':
        from xms.mf6.file_io.gwf.sto_reader import StoReader
        return StoReader()
    elif ftype == 'SWI6':
        from xms.mf6.file_io.griddata_package_reader import GriddataPackageReader
        return GriddataPackageReader(ftype=ftype)
    elif ftype == 'TAB6':
        from xms.mf6.file_io.gwf.tab_reader import TabReader
        return TabReader()
    elif ftype == 'TAS6':
        from xms.mf6.file_io.gwf.time_array_series_reader import TimeArraySeriesReader
        return TimeArraySeriesReader()
    elif ftype == 'TS6':
        from xms.mf6.file_io.gwf.time_series_reader import TimeSeriesReader
        return TimeSeriesReader()
    elif ftype == 'TVA6':
        return None
    elif ftype == 'UZF6':
        from xms.mf6.file_io.gwf.uzf_reader import UzfReader
        return UzfReader()
    elif ftype == 'VSC6':
        from xms.mf6.file_io.gwf.vsc_reader import VscReader
        return VscReader()
    elif ftype == 'WEL6':
        from xms.mf6.file_io.package_reader import PackageReader
        return PackageReader(ftype=ftype)
    elif ftype == 'ZONE6':
        from xms.mf6.file_io.gwf.zone_reader import ZoneReader
        return ZoneReader()

    # GWT
    elif ftype == 'ADV6':
        from xms.mf6.file_io.options_only_reader import OptionsOnlyReader
        return OptionsOnlyReader(ftype=ftype)
    elif ftype == 'CNC6':
        from xms.mf6.file_io.package_reader import PackageReader
        return PackageReader(ftype=ftype)
    elif ftype == 'DSP6':
        from xms.mf6.file_io.griddata_package_reader import GriddataPackageReader
        return GriddataPackageReader(ftype=ftype)
    elif ftype == 'FMI6':
        from xms.mf6.file_io.gwt.fmi_reader import FmiReader
        return FmiReader()
    elif ftype == 'GWT6':
        from xms.mf6.file_io.gwt.gwt_reader import GwtReader
        return GwtReader()
    elif ftype == 'IST6':
        from xms.mf6.file_io.griddata_package_reader import GriddataPackageReader
        return GriddataPackageReader(ftype=ftype)
    elif ftype == 'LKT6':
        from xms.mf6.file_io.advanced_transport_reader import AdvancedTransportReader
        return AdvancedTransportReader(ftype=ftype)
    elif ftype == 'MDT6':
        from xms.mf6.file_io.griddata_package_reader import GriddataPackageReader
        return GriddataPackageReader(ftype=ftype)
    elif ftype == 'MST6':
        from xms.mf6.file_io.griddata_package_reader import GriddataPackageReader
        return GriddataPackageReader(ftype=ftype)
    elif ftype == 'MVT6':
        from xms.mf6.file_io.options_only_reader import OptionsOnlyReader
        return OptionsOnlyReader(ftype=ftype)
    elif ftype == 'MWT6':
        from xms.mf6.file_io.advanced_transport_reader import AdvancedTransportReader
        return AdvancedTransportReader(ftype=ftype)
    elif ftype == 'SFT6':
        from xms.mf6.file_io.advanced_transport_reader import AdvancedTransportReader
        return AdvancedTransportReader(ftype=ftype)
    elif ftype == 'SRC6':
        from xms.mf6.file_io.package_reader import PackageReader
        return PackageReader(ftype=ftype)
    elif ftype == 'SSM6':
        from xms.mf6.file_io.gwt.ssm_reader import SsmReader
        return SsmReader()
    elif ftype == 'UZT6':
        from xms.mf6.file_io.advanced_transport_reader import AdvancedTransportReader
        return AdvancedTransportReader(ftype=ftype)

    # GWE
    elif ftype == 'CND6':
        from xms.mf6.file_io.griddata_package_reader import GriddataPackageReader
        return GriddataPackageReader(ftype=ftype)
    elif ftype == 'CTP6':
        from xms.mf6.file_io.package_reader import PackageReader
        return PackageReader(ftype=ftype)
    elif ftype == 'GWE6':
        from xms.mf6.file_io.gwe.gwe_reader import GweReader
        return GweReader()
    elif ftype == 'ESL6':
        from xms.mf6.file_io.package_reader import PackageReader
        return PackageReader(ftype=ftype)
    elif ftype == 'EST6':
        from xms.mf6.file_io.griddata_package_reader import GriddataPackageReader
        return GriddataPackageReader(ftype=ftype)
    elif ftype == 'LKE6':
        from xms.mf6.file_io.advanced_transport_reader import AdvancedTransportReader
        return AdvancedTransportReader(ftype=ftype)
    elif ftype == 'MVE6':
        from xms.mf6.file_io.options_only_reader import OptionsOnlyReader
        return OptionsOnlyReader(ftype=ftype)
    elif ftype == 'MWE6':
        from xms.mf6.file_io.advanced_transport_reader import AdvancedTransportReader
        return AdvancedTransportReader(ftype=ftype)
    elif ftype == 'SFE6':
        from xms.mf6.file_io.advanced_transport_reader import AdvancedTransportReader
        return AdvancedTransportReader(ftype=ftype)
    elif ftype == 'UZE6':
        from xms.mf6.file_io.advanced_transport_reader import AdvancedTransportReader
        return AdvancedTransportReader(ftype=ftype)

    # PRT
    elif ftype == 'PRT6':
        pass
    elif ftype == 'MIP6':
        pass
    elif ftype == 'PRP6':
        pass

    else:
        return None


# @staticmethod  # noqa: C901  (too complex)
def get_writer(ftype):
    """Returns the writer for the ftype.

    Imports are embedded in the if-else block to avoid circular
    dependencies encountered when listing them all at the top of the file.

    Args:
        ftype (str): The file type used in the model name file (e.g. 'WEL6')

    Returns:
        writer: objects that will write a modflow file
    """
    # Simulation
    if ftype == 'MFSIM6':
        from xms.mf6.file_io.mfsim_writer import MfsimWriter
        return MfsimWriter()

    # Simulation children
    if ftype == 'IMS6':
        from xms.mf6.file_io.gwf.ims_writer import ImsWriter
        return ImsWriter()
    elif ftype == 'TDIS6':
        from xms.mf6.file_io.tdis_writer import TdisWriter
        return TdisWriter()
    elif ftype == 'IMS6':
        from xms.mf6.file_io.gwf.ims_writer import ImsWriter
        return ImsWriter()
    elif ftype == 'EMS6':
        from xms.mf6.file_io.gwf.ems_writer import EmsWriter
        return EmsWriter()

    # Exchanges
    elif ftype in {'GWF6-GWF6', 'GWT6-GWT6', 'GWE6-GWE6'}:
        from xms.mf6.file_io.exchanges.gwx_gwx_writer import GwxGwxWriter
        return GwxGwxWriter()
    elif ftype in {'GWF6-GWT6', 'GWF6-GWE6', 'GWF6-PRT6'}:
        from xms.mf6.file_io.exchanges.exchange_writer_base import ExchangeWriterBase
        return ExchangeWriterBase()

    # GWF
    elif ftype == 'ATS6':
        from xms.mf6.file_io.ats_writer import AtsWriter
        return AtsWriter()
    elif ftype == 'BUY6':
        from xms.mf6.file_io.gwf.buy_writer import BuyWriter
        return BuyWriter()
    elif ftype == 'CHD6':
        from xms.mf6.file_io.list_package_writer import ListPackageWriter
        return ListPackageWriter()
    elif ftype == 'CSUB6':
        from xms.mf6.file_io.gwf.csub_writer import CsubWriter
        return CsubWriter()
    elif ftype == 'DIS6':
        from xms.mf6.file_io.dis_writer import DisWriter
        return DisWriter()
    elif ftype == 'DISU6':
        from xms.mf6.file_io.disu_writer import DisuWriter
        return DisuWriter()
    elif ftype == 'DISV6':
        from xms.mf6.file_io.disv_writer import DisvWriter
        return DisvWriter()
    elif ftype == 'DRN6':
        from xms.mf6.file_io.list_package_writer import ListPackageWriter
        return ListPackageWriter()
    elif ftype == 'EVT6':
        from xms.mf6.file_io.gwf.evt_writer import EvtWriter
        return EvtWriter()
    elif ftype == 'GHB6':
        from xms.mf6.file_io.list_package_writer import ListPackageWriter
        return ListPackageWriter()
    elif ftype == 'GNC6':
        from xms.mf6.file_io.gwf.gnc_writer import GncWriter
        return GncWriter()
    elif ftype == 'GWF6':
        from xms.mf6.file_io.gwf.gwf_writer import GwfWriter
        return GwfWriter()
    elif ftype == 'HFB6':
        from xms.mf6.file_io.gwf.hfb_writer import HfbWriter
        return HfbWriter()
    elif ftype == 'IC6':
        from xms.mf6.file_io.ic_writer import IcWriter
        return IcWriter()
    elif ftype == 'LAK6':
        from xms.mf6.file_io.gwf.lak_writer import LakWriter
        return LakWriter()
    elif ftype == 'MAW6':
        from xms.mf6.file_io.gwf.maw_writer import MawWriter
        return MawWriter()
    elif ftype == 'MVR6':
        from xms.mf6.file_io.gwf.mvr_writer import MvrWriter
        return MvrWriter()
    elif ftype == 'NPF6':
        from xms.mf6.file_io.gwf.npf_writer import NpfWriter
        return NpfWriter()
    elif ftype == 'OBS6':
        from xms.mf6.file_io.obs_writer import ObsWriter
        return ObsWriter()
    elif ftype == 'OC6':
        from xms.mf6.file_io.gwf.oc_writer import OcWriter
        return OcWriter()
    elif ftype == 'POBS6':
        from xms.mf6.file_io.pest.pest_obs_writer import PestObsWriter
        return PestObsWriter()
    elif ftype == 'RCH6':
        from xms.mf6.file_io.gwf.rch_writer import RchWriter
        return RchWriter()
    elif ftype == 'RIV6':
        from xms.mf6.file_io.list_package_writer import ListPackageWriter
        return ListPackageWriter()
    elif ftype == 'SFR6':
        from xms.mf6.file_io.gwf.sfr_writer import SfrWriter
        return SfrWriter()
    elif ftype == 'STO6':
        from xms.mf6.file_io.gwf.sto_writer import StoWriter
        return StoWriter()
    elif ftype == 'SWI6':
        from xms.mf6.file_io.griddata_package_writer import GriddataPackageWriter
        return GriddataPackageWriter()
    elif ftype == 'TAB6':
        from xms.mf6.file_io.gwf.tab_writer import TabWriter
        return TabWriter()
    elif ftype == 'TAS6':
        from xms.mf6.file_io.gwf.time_array_series_writer import TimeArraySeriesWriter
        return TimeArraySeriesWriter()
    elif ftype == 'TS6':
        from xms.mf6.file_io.gwf.time_series_writer import TimeSeriesWriter
        return TimeSeriesWriter()
    elif ftype == 'TVA6':
        from xms.mf6.file_io.gwf.array_package_writer import ArrayPackageWriter
        return ArrayPackageWriter()
    elif ftype == 'UZF6':
        from xms.mf6.file_io.gwf.uzf_writer import UzfWriter
        return UzfWriter()
    elif ftype == 'VSC6':
        from xms.mf6.file_io.gwf.vsc_writer import VscWriter
        return VscWriter()
    elif ftype == 'WEL6':
        from xms.mf6.file_io.list_package_writer import ListPackageWriter
        return ListPackageWriter()
    elif ftype == 'ZONE6':
        from xms.mf6.file_io.gwf.zone_writer import ZoneWriter
        return ZoneWriter()

    # GWT
    elif ftype == 'ADV6':
        from xms.mf6.file_io.options_only_writer import OptionsOnlyWriter
        return OptionsOnlyWriter()
    elif ftype == 'CNC6':
        from xms.mf6.file_io.list_package_writer import ListPackageWriter
        return ListPackageWriter()
    elif ftype == 'DSP6':
        from xms.mf6.file_io.griddata_package_writer import GriddataPackageWriter
        return GriddataPackageWriter()
    elif ftype == 'FMI6':
        from xms.mf6.file_io.gwt.fmi_writer import FmiWriter
        return FmiWriter()
    elif ftype == 'GWT6':
        from xms.mf6.file_io.gwt.gwt_writer import GwtWriter
        return GwtWriter()
    elif ftype == 'IST6':
        from xms.mf6.file_io.griddata_package_writer import GriddataPackageWriter
        return GriddataPackageWriter()
    elif ftype == 'LKT6':
        from xms.mf6.file_io.advanced_transport_writer import AdvancedTransportWriter
        return AdvancedTransportWriter()
    elif ftype == 'MDT6':
        from xms.mf6.file_io.griddata_package_writer import GriddataPackageWriter  # pragma no cover
        return GriddataPackageWriter()
    elif ftype == 'MST6':
        from xms.mf6.file_io.griddata_package_writer import GriddataPackageWriter
        return GriddataPackageWriter()
    elif ftype == 'MVT6':
        from xms.mf6.file_io.options_only_writer import OptionsOnlyWriter
        return OptionsOnlyWriter()
    elif ftype == 'MWT6':
        from xms.mf6.file_io.advanced_transport_writer import AdvancedTransportWriter
        return AdvancedTransportWriter()
    elif ftype == 'SFT6':
        from xms.mf6.file_io.advanced_transport_writer import AdvancedTransportWriter
        return AdvancedTransportWriter()
    elif ftype == 'SRC6':
        from xms.mf6.file_io.list_package_writer import ListPackageWriter
        return ListPackageWriter()
    elif ftype == 'SSM6':
        from xms.mf6.file_io.gwt.ssm_writer import SsmWriter
        return SsmWriter()
    elif ftype == 'UZT6':
        from xms.mf6.file_io.advanced_transport_writer import AdvancedTransportWriter
        return AdvancedTransportWriter()

    # GWE
    elif ftype == 'CND6':
        from xms.mf6.file_io.griddata_package_writer import GriddataPackageWriter
        return GriddataPackageWriter()
    elif ftype == 'CTP6':
        from xms.mf6.file_io.list_package_writer import ListPackageWriter
        return ListPackageWriter()
    elif ftype == 'GWE6':
        from xms.mf6.file_io.gwe.gwe_writer import GweWriter
        return GweWriter()
    elif ftype == 'ESL6':
        from xms.mf6.file_io.list_package_writer import ListPackageWriter
        return ListPackageWriter()
    elif ftype == 'EST6':
        from xms.mf6.file_io.griddata_package_writer import GriddataPackageWriter
        return GriddataPackageWriter()
    elif ftype == 'LKE6':
        from xms.mf6.file_io.advanced_transport_writer import AdvancedTransportWriter
        return AdvancedTransportWriter()
    elif ftype == 'MVE6':
        from xms.mf6.file_io.options_only_writer import OptionsOnlyWriter
        return OptionsOnlyWriter()
    elif ftype == 'MWE6':
        from xms.mf6.file_io.advanced_transport_writer import AdvancedTransportWriter
        return AdvancedTransportWriter()
    elif ftype == 'SFE6':
        from xms.mf6.file_io.advanced_transport_writer import AdvancedTransportWriter
        return AdvancedTransportWriter()
    elif ftype == 'UZE6':
        from xms.mf6.file_io.advanced_transport_writer import AdvancedTransportWriter
        return AdvancedTransportWriter()

    # PRT
    elif ftype == 'PRT6':
        pass
    elif ftype == 'MIP6':
        pass
    elif ftype == 'PRP6':
        pass

    else:
        return None


# @staticmethod
def reader_from_ftype(ftype):
    """Gets the package reader from the ftype and sets the grid_dimension and nper variables.

    Args:
        ftype (str): The file type used in the model name file (e.g. 'WEL6')

    Returns:
        reader: objects that will read a modflow file
    """
    reader = get_reader(ftype)
    return reader


def writer_from_ftype(ftype, writer_options=None):
    """Gets the package reader from the ftype and sets the grid_dimension and nper variables.

    Args:
        ftype (str): The file type used in the model name file (e.g. 'WEL6')
        writer_options (WriterOptions) : class with various writer options

    Returns:
        writer: objects that will write a modflow file
    """
    writer = get_writer(ftype.upper())
    if writer:
        if writer_options:
            writer._writer_options = copy.deepcopy(writer_options)
        return writer
    return None
