"""ModelReaderBase class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass
import os
from pathlib import Path
import shlex
import sys
from typing import Container

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.api.tree import TreeNode
from xms.core.filesystem import filesystem as fs
from xms.data_objects.parameters import UGrid as DoGrid
from xms.testing.type_aliases import Pathlike

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.grid_info import DisEnum
from xms.mf6.data.model_data_base import ModelDataBase
from xms.mf6.file_io import io_factory, io_util
from xms.mf6.file_io.package_reader import PackageReader
from xms.mf6.file_io.text_file_context_manager import TextFileContextManager
from xms.mf6.gui import gui_util
from xms.mf6.misc import log_util
from xms.mf6.misc.settings import Settings

_PNAME_MAX_CHAR = 16  # "PNAME is restricted to 16 characters."


def _read_projection_file(dis_filepath: Path | str) -> str:
    """Read the .prj file next to the dis* package file, if one exists, and return the contents of the file (wkt).

    Args:
        dis_filepath: Filepath of dis* file.

    Returns:
        See description.
    """
    wkt = ''
    prj_filepath = f'{str(dis_filepath)}.prj'
    if Path(prj_filepath).is_file():
        with open(prj_filepath, 'r') as file:
            wkt = file.read()
    return wkt


@dataclass
class Package:
    """Represents a line in the PACKAGES block in the model name file (<ftype> <fname> [<pname>])."""
    ftype: str = ''
    fname: str = ''
    pname: str = ''


def packages_from_model_name_file(model_filename: Pathlike, ftypes: str | Container[str], first: bool) -> list[Package]:
    """Reads the name file and returns the packages of given ftype, or all packages if ftype == ''.

    The fnames are resolved to full paths.

    Args:
        model_filename: The name file.
        ftypes: The ftype as a string or a collection of strings.
        first: If True, search will stop when the first match is found.

    Returns:
        List of packages.
    """
    if not isinstance(ftypes, Container):  # Make a set out of a single string, if that's what was passed
        ftypes = {ftypes}
    packages: list[Package] = []
    with TextFileContextManager(model_filename) as fp:
        while True:
            line = fp.readline()
            if not line:
                break

            if io_util.is_begin_line(line):
                if io_util.is_begin_x_line(line, 'PACKAGES'):
                    model_dir = os.path.dirname(model_filename)
                    while True:
                        line2 = fp.readline()
                        if not line2:
                            break

                        if io_util.is_end_line(line2):
                            break
                        else:
                            words = line2.split()
                            ftype = words[0].upper()
                            if not ftypes or ftype in ftypes:
                                fname = fs.resolve_relative_path(model_dir, words[1])
                                pname = '' if len(words) < 3 else words[2].strip('"\'‘’“”')
                                packages.append(Package(ftype, fname, pname))
                                if first:
                                    break
                    break
                else:
                    io_util.skip_block(fp)
    return packages


def dis_from_model_name_file(model_filename: Pathlike) -> DisEnum:
    """Reads the name file and returns which DIS package is being used: DIS, DISV, or DISU.

    Note that this should only be used for tests because in production, the mfsim.nam file
    may be out of date and the project explorer tree should be used instead.

    Args:
        model_filename: File path to GWF name file.

    Returns:
        (tuple): tuple containing:
            - (DisEnum): The DIS package in use.
            - (str): The dis filename.
    """
    packages: list[Package] = packages_from_model_name_file(model_filename, {'DIS6', 'DISV6', 'DISU6'}, first=True)
    if not packages:
        return DisEnum.END
    dis_enum_from_ftype = {'DIS6': DisEnum.DIS, 'DISV6': DisEnum.DISV, 'DISU6': DisEnum.DISU}
    return dis_enum_from_ftype[packages[0].ftype]


def list_file_from_model_name_file(model_filename):
    """Reads part of the model name file and returns the LIST filename if found.

    Args:
        model_filename (str): File path of OC file.

    Returns:
        See description.
    """
    list_filename = os.path.splitext(model_filename)[0] + '.lst'
    with open(model_filename, 'r') as file:
        for line in file:
            if io_util.is_begin_x_line(line, 'OPTIONS'):
                for line2 in file:
                    line2 = line2.strip()
                    if line2.upper().startswith('LIST'):
                        list_filename = line2.split()[1]
                        break
                    elif io_util.is_end_line(line2):
                        break
                break
    if not os.path.isabs(list_filename):
        list_filename = os.path.join(os.path.dirname(model_filename), list_filename)
        list_filename = os.path.normpath(list_filename)
    return list_filename


def _read_grid(filename: Path | str, data: ModelDataBase) -> None:
    """When testing, if a 'ugrid.xmc' file exists, a dogrid is created and associated with the model.

    This is only done when testing and lets us get grids that would otherwise be obtained from XMS via a query.

    Args:
        filename: Package filename.
        data: Package data.
    """
    ugrid_file = Path(filename).with_name('ugrid.xmc')
    if ugrid_file.exists():
        # Get the grid uuid. Read just the UUID from the file instead of using read_grid_from_file()
        ugrid_uuid = ''
        with open(ugrid_file, 'r') as file:
            for line in file:
                if line.startswith('UUID'):
                    ugrid_uuid = line.split()[1]
                    break
        if not ugrid_uuid:
            return

        # Get model uuid
        if data.tree_node:
            model_uuid = data.tree_node.uuid
        else:
            # Get model uuid from parent directory name, if possible
            model_uuid = io_util.uuid_from_path(filename)
        if model_uuid:
            dogrid = DoGrid(str(ugrid_file))
            dogrid.uuid = ugrid_uuid
            data.mfsim.xms_data.set_dogrid(model_uuid, dogrid)


class ModelReaderBase(PackageReader):
    """Reads a GWF model file."""
    def __init__(self, ftype):
        """Initializes the class.

        Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
        """
        super().__init__(ftype=ftype)
        self._package_lines = []
        self._stop_after_block = 'PACKAGES'
        self._model_node = None  # If set, the GWF package is created using info from the Project Explorer
        self.node_to_data = {}  # Dict filled in _read_from_tree so that packages can later be found from tree nodes
        self._log = log_util.get_logger()
        self._pnames: set[str] = set()  # Used to ensure unique pnames (which become tree names)

    def _read_package(self, words):
        """Reads the package.

        Args:
            words (list of str): ftype, fname and maybe pname.

        Return:
            The package data object.
        """
        ftype = words[0].upper()
        reader = io_factory.reader_from_ftype(ftype)
        if reader:
            filename = fs.resolve_relative_path(os.path.dirname(self._data.filename), words[1])
            data = reader.read(filename, mfsim=self._data.mfsim, model=self._data)
            if not data:
                self._log.warning(f'Unable to read file {filename}.')
                return
            data.ftype = ftype

            # Set pname
            if len(words) > 2:
                data.pname = words[2].strip('"\'‘’“”').replace(' ', '-')
            else:
                data.pname = gui_util.unique_name_no_spaces(self._pnames, ftype[:-1], '-')
            self._pnames = data.pname
            return data
        return None

    def _read_dis(self):
        """Reads the DIS, DISV, or DISU package."""
        for words in self._package_lines:
            if words[0].upper() in ['DIS6', 'DISV6', 'DISU6']:
                data = self._read_package(words)
                if data:
                    self._data.add_package(data)
                    self._data.projection_wkt = _read_projection_file(data.filename)
                # Remove DIS package so that it doesn't get read again later
                self._package_lines.remove(words)  # noqa B038 editing a loop's mutable iterable
                return

    def _on_end_block(self, block_name):
        """Called when an END [block] line is found.

        Args:
            block_name (str): Name of the current block.
        """
        if block_name != 'PACKAGES' or self._model_node:
            return

        # First read the DIS package so other package will have the grid info
        self._read_dis()

        for words in self._package_lines:
            data = self._read_package(words)
            if data:
                self._data.add_package(data)

    def _read_packages(self, line):
        """Reads the packages.

        Args:
            line (str): A line from the file.
        """
        if self._model_node:
            self._read_packages_from_tree()
            return True  # Return True to tell reader that we read the whole block

        # Use shlex to handle quoted strings
        words = shlex.split(line, posix="win" not in sys.platform)
        if len(words) < 2:
            self._log.warning(
                f'Invalid format in file {self._data.filename} on or around line'
                f' {self._line_number}.'
            )
            return

        if not io_factory.reader_exists(words[0].upper()):
            self._log.warning(f'No reader found for ftype {words[0].upper()} in file {self._data.filename}.')
            return

        # Store the package lines and process them after reading the block
        self._package_lines.append(words)

    def _read_packages_from_tree(self):
        """Reads the GWF packages as defined by the Project Explorer, not the name file."""
        self.node_to_data.clear()
        self._data.mname = self._model_node.name
        self._data.filename = self._model_node.main_file

        # Must read DIS* package first
        for child in self._model_node.children:
            ftype = child.unique_name
            if ftype in ['DIS6', 'DISV6', 'DISU6']:
                self._read_child_package(child)
                break

        # Now read everything except the DIS* package
        for child in self._model_node.children:
            ftype = child.unique_name
            if ftype and ftype not in ['DIS6', 'DISV6', 'DISU6']:
                self._read_child_package(child)

    def _read_child_package(self, child: TreeNode) -> None:
        """Reads the package associated with the child node.

        Args:
            child: A tree node of the Project Explorer.
        """
        ftype = child.unique_name
        main_file = child.main_file
        reader = io_factory.reader_from_ftype(ftype)
        a_based = ftype in data_util.readasarrays_ftypes(with_the_a=True)
        data = reader.read(main_file, mfsim=self._data.mfsim, model=self._data, tree_node=child, array_based=a_based)
        if data:
            data.pname = child.name
            self._data.add_package(data)
            self.node_to_data[child.uuid] = data

    @override
    def read_settings(self) -> None:
        """Reads the package settings file if it exists."""
        if os.path.exists(Settings.settings_filename(self._data.filename)):
            settings = Settings.read_settings(self._data.filename)
            if 'PACKAGES' in settings:
                packages_by_uuid = self._build_packages_by_uuid_dict()
                self._assign_fname_to_packages(packages_by_uuid, settings['PACKAGES'])

    def _assign_fname_to_packages(self, packages_by_uuid, package_fnames):
        """Uses the packages_by_uuid dict to assign fnames to the packages.

        Args:
            packages_by_uuid (dict[str, package]): Dict of packages keyed by their uuid.
            package_fnames (dict[str, str]): Dict of package uuids keyed by fnames.
        """
        for fname, uuid in package_fnames.items():
            if uuid in packages_by_uuid:
                packages_by_uuid[uuid].fname = fname

    def _build_packages_by_uuid_dict(self):
        """Creates and returns a dict of packages keyed by their uuid.

        Returns:
            (dict[str, package]): See description.
        """
        packages_by_uuid = {}
        for package in self._data.packages:
            if package.tree_node:
                packages_by_uuid[package.tree_node.uuid] = package
        return packages_by_uuid

    def read(self, filename, model_node=None, **kwargs):
        """Reads the GWF name file.

        Args:
            filename (str): File path of GWF name file.
            model_node (TreeNode): Optionally the model item from the Project Explorer. If not none, packages list is
             created from the Project Explorer, not the GWF name file.
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            mfsim (MfsimData): The simulation.
        """
        self._model_node = model_node
        data = super().read(filename, **kwargs)
        if XmEnv.xms_environ_running_tests() == 'TRUE':
            _read_grid(filename, data)  # This is only done for testing purposes
        return data
