"""DisuDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QLabel, QTabWidget, QVBoxLayout, QWidget

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.block import Block
from xms.mf6.gui import gui_util
from xms.mf6.gui.dis_dialog_base import create_plain_text_edit, fill_edt_with_file
from xms.mf6.gui.disv_dialog import DisvDialog


class DisuDialog(DisvDialog):
    """A dialog used for the DISU package."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'DIMENSIONS', 'GRIDDATA', 'CONNECTIONDATA', 'VERTICES', 'CELL2D']
        self.default_sections = ['GRIDDATA']

    def setup_dimensions(self):
        """Sets up the dimensions section."""
        name = 'DIMENSIONS'
        self.add_group_box_to_scroll_area(name)

        # NODES
        self.uix[name]['txt_nodes'] = QLabel('Number of nodes (NODES):')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_nodes'])

        # NJA
        self.uix[name]['txt_nja'] = QLabel('Number of connections and nodes (NJA):')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_nja'])

        # NVERT
        self.uix[name]['txt_nvert'] = QLabel('Number of vertices (NVERT):')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_nvert'])

        self.uix[name]['txt_nodes'].setText(f'Number of nodes (NODES): {self.dlg_input.data.grid_info().nodes}')
        self.uix[name]['txt_nja'].setText(
            f'Number of connections and nodes (NJA): {self.dlg_input.data.grid_info().nja}'
        )
        self.uix[name]['txt_nvert'].setText(f'Number of vertices (NVERT): {self.dlg_input.data.grid_info().nvert}')

    def create_connectiondata_tabs(self):
        """Creates the tabs for the CONNECTIONDATA block."""
        block_name = 'CONNECTIONDATA'
        connectiondata: Block = self.dlg_input.data.block(block_name)
        for i, name in enumerate(connectiondata.names):
            widgets = {}
            page = QWidget()
            vlayout = QVBoxLayout()

            # Create text edit field
            widgets['edt_text'] = create_plain_text_edit()
            vlayout.addWidget(widgets['edt_text'])

            page.setLayout(vlayout)
            self.uix[block_name]['tab_widget'].addTab(page, name)
            self.uix[block_name]['tab_widget'].setTabToolTip(
                i, gui_util.wrap_tip(self.dlg_input.data.get_tool_tip(name))
            )
            page.setUpdatesEnabled(True)  # To enable signals.

            array = connectiondata.array(name)
            if array:
                array_layer = array.layer(0)
                if array_layer.storage == 'CONSTANT':
                    txt = f'CONSTANT {array_layer.factor}'
                    file_name = ''
                else:
                    txt = ''
                    file_name = array_layer.external_filename
                # widget_builder.fill_edt_with_file(file_name, widgets['txt_edit'], txt)
                fill_edt_with_file(file_name, widgets['edt_text'], txt)

    def setup_connectiondata(self):
        """Sets up the CONNECTIONDATA block."""
        name = 'CONNECTIONDATA'
        self.add_group_box_to_scroll_area(name)

        # read only message
        txt = 'CONNECTIONDATA is defined by the grid and is not editable.'
        self.uix[name]['txt_read_only'] = QLabel(txt)
        self.uix[name]['txt_read_only'].setStyleSheet('font-weight: bold; color: black')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_read_only'])
        self.uix[name]['tab_widget'] = QTabWidget()
        self.uix[name]['layout'].addWidget(self.uix[name]['tab_widget'])

        self.create_connectiondata_tabs()

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'CONNECTIONDATA':
            self.setup_connectiondata()
        else:
            super().setup_section(section_name)
