"""GetIntDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import (QDialog, QDialogButtonBox, QHBoxLayout, QLabel, QSpinBox, QVBoxLayout)

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.mf6.misc.settings import Settings


class GetIntDialog(XmsDlg):
    """A simple dialog to get an integer value from the user."""
    def __init__(self, title, label, value, minimum, maximum, parent, main_file=''):
        """Initializes the class, sets up the ui.

        Args:
            title (str): Title of the dialog
            label (str): Label next to the spin box
            value (int): Initial value in the spin box
            minimum (int): Spin box minimum
            maximum (int): Spin box maximum
            parent (Something derived from QWidget): The parent window.
            main_file (str): If given, will attempt to read the int from the last time.
        """
        super().__init__(parent, 'xms.mf6.gui.get_int_dialog')

        if main_file and value is None:
            value = int(Settings.get(main_file, 'GET_INT_VALUE', 10))

        self.value = value
        self.main_file = main_file
        self.setWindowTitle(title)
        vlayout = QVBoxLayout(self)
        hlayout = QHBoxLayout()
        txt = QLabel(label, self)
        self.spn = QSpinBox(self)
        self.spn.setRange(minimum, maximum)
        self.spn.setValue(value)
        hlayout.addWidget(txt)
        hlayout.addWidget(self.spn)
        vlayout.addLayout(hlayout)
        self.btn_box = QDialogButtonBox(self)
        self.btn_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Ok)
        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        vlayout.addWidget(self.btn_box)

    def accept(self):
        """Called when OK button is clicked."""
        self.value = self.spn.value()
        if self.main_file:
            Settings.set(self.main_file, 'GET_INT_VALUE', self.value)
        super().accept()


def run(title, label, value, minimum, maximum, parent, main_file=''):
    """Runs the GetIntDialog.

    Args:
        title (str): Title of the dialog
        label (str): Label next to the spin box
        value (int): Initial value in the spin box
        minimum (int): Spin box minimum
        maximum (int): Spin box maximum
        parent (Something derived from QWidget): The parent window.
        main_file (str): If given and value is None, attempts to read the int from the last time.

    Returns:
        (tuple): tuple containing:
            - (bool): True or False (for OK or Cancel).
            - (int): the integer value.
    """
    dialog = GetIntDialog(title, label, value, minimum, maximum, parent, main_file)
    if dialog.exec() == QDialog.Accepted:
        return True, dialog.value
    else:
        return False, 0
