"""CsubDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.gui import list_dialog
from xms.mf6.gui.data_tabs_widget import DataTabsWidget
from xms.mf6.gui.obs_dialog import ObsDialog
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog_base import PackageDialogBase
from xms.mf6.gui.widgets.griddata_widget import GriddataWidget


class CsubDialog(PackageDialogBase):
    """A dialog used for the CSUB package."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)
        self.griddata_widget: GriddataWidget | None = None
        self.data_tabs_widget: DataTabsWidget | None = None
        self.setup_ui()

    @override
    def setup_ui(self) -> None:
        """Set up everything dealing with sections."""
        self.griddata_widget = GriddataWidget(self.dlg_input, read_only_tabs=[], parent=self)
        self.data_tabs_widget = DataTabsWidget(self.dlg_input, self)
        super().setup_ui()

    @override
    def clear_sections(self) -> None:
        """Clear all section widgets."""
        self.griddata_widget = None
        self.data_tabs_widget = None
        super().clear_sections()

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'GRIDDATA', 'DATA']
        self.default_sections = ['DATA']

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'GRIDDATA':
            self.setup_griddata_section()
        elif section_name == 'DATA':
            self.setup_data_section()
        else:
            super().setup_section(section_name)

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def on_chk_boundnames(self, checked):
        """Warns user that this will add or remove the boundname column and lets them cancel.

        Args:
            checked (bool): True if checkbox is checked.
        """
        self.dlg_input.data.options_block.set(key='BOUNDNAMES', on=(checked == Qt.Checked), value=None)
        self.data_tabs_widget.change_list_block_columns(self.dlg_input.data.block_with_boundnames())

    def on_btn_obs6_filein(self):
        """Opens the dialog and updates the list of observation files."""
        list_dialog.run_list_dialog_for_filein(
            ftype='OBS6',
            parent=self,
            options_block=self.dlg_input.data.options_block,
            edit_func=ObsDialog.run_dialog_on_file
        )

    def setup_griddata_section(self):
        """Sets up the GRIDDATA section."""
        name = 'GRIDDATA'
        self.add_group_box_to_scroll_area(name)

        self.uix[name]['griddata_widget'] = self.griddata_widget
        self.uix[name]['layout'].addWidget(self.griddata_widget)
        self.griddata_widget.setup()

    def setup_data_section(self):
        """Sets up the DATA section."""
        name = 'DATA'
        self.add_group_box_to_scroll_area(name)

        self.uix[name]['data_tabs_widget'] = self.data_tabs_widget
        self.uix[name]['layout'].addWidget(self.data_tabs_widget)
        self.data_tabs_widget.setup()

    # @overrides
    def get_table_widget(self, block):
        """Returns the table view from the given block.

        Args:
            block: The block.
        """
        return self.data_tabs_widget.get_table_widget(block)

    @override
    def widgets_to_data(self) -> None:
        """Get info from widgets and store it in dlg_input.data."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.data_tabs_widget.accept()
            self.save_list_blocks_to_temp()

    def reject(self):
        """Called when the Cancel button is clicked."""
        self.data_tabs_widget.reject()
        super().reject()
