"""FlowPackageDataFinder class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import ComponentItem
from xms.api.tree import tree_util
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg

# 4. Local modules
from xms.mf6.file_io import io_factory
from xms.mf6.gui import gui_util
from xms.mf6.misc import util


class FlowPackageDataFinder:
    """Opens dialog to select advanced flow package, then extracts data from PACKAGEDATA block."""
    def __init__(self, flow_ftype, dlg_input, parent=None):
        """Initializes the class.

        Args:
            flow_ftype (str): The file type used in the model name file (e.g. 'WEL6')
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        self.flow_ftype = flow_ftype
        self.dlg_input = dlg_input
        self.parent = parent

    def _selectable_callback(self, node):
        """Used by TreeItemSelectorDlg to determine what tree nodes are selectable.

        Args:
            node (TreeNode): A tree node.

        Returns:
            (bool): If the node is selectable.
        """
        return node.unique_name == self.flow_ftype

    def _filter_condition(self, node):
        """Condition used by util.filter_tree.

        Args:
            node (TreeNode): A node.

        Returns:
            (bool): True if the node should be kept.
        """
        return type(node.data) is ComponentItem and node.unique_name == self.flow_ftype

    def _select_item_from_dialog(self, trimmed_tree):
        """Opens the tree selector dialog so user can pick the tree model.

        Args:
            trimmed_tree (TreeNode): The tree to show in the dialog.

        Returns:
            (str): Uuid of the selected tree node.
        """
        dialog = TreeItemSelectorDlg(
            title=f'Select {self.flow_ftype}',
            target_type=ComponentItem,
            pe_tree=trimmed_tree,
            override_icon=gui_util.get_icon_path,
            parent=self.parent,
            selectable_callback=self._selectable_callback
        )
        if dialog.exec() != QDialog.Accepted:
            return None
        flow_node_uuid = dialog.get_selected_item_uuid()
        if not flow_node_uuid:
            return None
        return flow_node_uuid

    def _extract_data(self, data):
        """Returns data needed for transport PACKAGEDATA block by reading flow PACKAGEDATA block.

        Args:
            data: The flow package.

        Returns:
            (dict): The data.
        """
        column_names, column_types, _ = data.get_column_info('PACKAGEDATA')
        flow_package_data = {column_names[0]: [], 'STRT': []}
        df = data.read_csv_file_into_dataframe(
            'PACKAGEDATA', data.list_blocks['PACKAGEDATA'], column_names, column_types
        )
        flow_package_data[column_names[0]] = df[column_names[0]].tolist()
        flow_package_data['STRT'] = [0.0] * len(flow_package_data[column_names[0]])
        return flow_package_data

    def _sim_node_from_descendant_node(self, flow_node):
        """Returns the simulation TreeNode given a descendant node.

        Args:
            flow_node (TreeNode): Some descendant of the sim.

        Returns:
            (TreeNode): The sim node.
        """
        flow_sim_node = flow_node.parent
        while flow_sim_node.item_typename != 'TI_DYN_SIM':
            flow_sim_node = flow_sim_node.parent
        return flow_sim_node

    def _get_flow_sim(self, flow_node):
        """Returns the flow simulation (MfsimData) given the flow package node.

        Args:
            flow_node (TreeNode): The flow package tree node.

        Returns:
            (MfsimData): The flow simulation.
        """
        flow_sim_node = self._sim_node_from_descendant_node(flow_node)
        flow_hidden_sim_node = self.dlg_input.query.item_with_uuid(
            item_uuid=flow_sim_node.uuid, model_name="MODFLOW 6", unique_name="Sim_Manager"
        )
        mfsim_nam = flow_hidden_sim_node.main_file
        if mfsim_nam != self.dlg_input.mfsim.filename:  # Flow sim is different than transport sim
            reader = io_factory.reader_from_ftype('MFSIM6')
            flow_mfsim = reader.read(mfsim_nam, flow_sim_node, self.dlg_input.query)
        else:  # Flow and transport sim are the same
            flow_mfsim = self.dlg_input.mfsim
        return flow_mfsim

    def find_data(self):
        """Finds and returns data in the flow model.

        Returns:
           (dict): The data.
        """
        flow_data = {}
        try:
            tree_copy = self.dlg_input.query.copy_project_tree()
            trimmed_tree = util.filter_tree(start_node=tree_copy, condition=self._filter_condition)
            if not trimmed_tree:
                return None

            flow_package_uuid = self._select_item_from_dialog(trimmed_tree)
            if not flow_package_uuid:
                return flow_data

            flow_node = tree_util.find_tree_node_by_uuid(self.dlg_input.query.project_tree, flow_package_uuid)
            if not flow_node:
                raise RuntimeError('Error getting package.')
            flow_mfsim = self._get_flow_sim(flow_node)
            data = flow_mfsim.package_from_filename(flow_node.main_file)
            if not data:
                raise RuntimeError('Error getting package.')
            if data.ftype != self.flow_ftype:
                raise RuntimeError('Item selected is not the correct type.')
            flow_data = self._extract_data(data)
        except Exception as error:
            message_box.message_with_ok(parent=self.parent, message=str(error), icon='Critical')

        return flow_data
