"""CellAdderArcs class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing_extensions import override

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.mapping.cell_adder import CellAdder


class CellAdderArcs(CellAdder):
    """Adds a cell BC."""
    def __init__(self, builder, cell_idx1, cell_idx2, record, period_rows: list, length: float, t_value_avg: float):
        """Initializes the class.

        Args:
            builder: The package builder.
            cell_idx1: Cell index.
            cell_idx2: Cell index.
            record: Shapefile record for the current arc.
            period_rows: List of rows for a stress period.
            length: Intersected length of the arc in the cell.
            t_value_avg: Average t_value for the cell
        """
        super().__init__(builder)
        self.set_record(cell_idx1, cell_idx2, record, period_rows)
        self._length = length
        self._t_value_avg = t_value_avg

    @override
    def _column_values_from_dicts(self, row_or_record, feature_id: int):
        """Returns the list of column values (a row) given row_or_record and map_import_info.

        Args:
            row_or_record: Can be either a record, if reading the shapefile, or a row, if reading the CSV file.
            feature_id (int): ID of current feature object.

        Returns:
            (list): See description.
        """
        column_values = []
        map_info_iter = iter(self._builder.map_import_info.items())
        for key, mapping in map_info_iter:
            if mapping and mapping.node2:  # Attribute at the nodes
                node_val0 = self._column_value(feature_id, key, row_or_record, None)
                node_val1 = self._column_value(feature_id, mapping.node2, row_or_record, None)
                value = node_val0 + ((node_val1 - node_val0) * self._t_value_avg)
                column_values.append(value)
                next(map_info_iter)  # Skip the next one since we already handled it
            else:  # Attribute at the arc
                column_values.append(self._column_value(feature_id, key, row_or_record, self._length))
        return column_values
