"""Utility functions used by other modules."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import bisect
import collections.abc
from difflib import SequenceMatcher
import math
import os
from pathlib import Path
import shutil
import statistics
from typing import Sequence
import uuid

# 2. Third party modules
import numpy as np
from PySide2.QtGui import (QIcon)

# 3. Aquaveo modules
from xms.api.tree import TreeNode
from xms.guipy.resources import resources_util
from xms.testing.type_aliases import Pathlike

# 4. Local modules

XM_NODATA = -9999999
"""A key value meaning the data should be ignored."""

max_int32 = 2147483647  # 2^31 − 1
"""The maximum 32 bit integer value"""

app_name = 'GMS'
"""Name of the XMS app"""

wiki_gms = 'https://www.xmswiki.com/wiki/GMS'
"""URL to the wiki"""

wiki_mf6 = 'https://www.xmswiki.com/wiki/GMS:MODFLOW_6'
"""URL of the default wiki page to go to if there's not a better one"""

wiki_dialog_help = 'https://www.xmswiki.com/wiki/GMS:GMS_99.99_Dialog_Help'
"""URL to the wiki"""

PRT_INTERFACE = False
"""Used to hide the PRT interface until it is ready."""

_app_icon: QIcon | None = None  # We save the app icon to make testing easier


def get_app_icon() -> QIcon:
    """Returns the application (GMS) icon.

    We use the module variable _app_icon to make testing easier (see test_on_btn_edit() in test_list_dialog.py).
    """
    global _app_icon
    if _app_icon is None:
        _app_icon = QIcon(resources_util.get_resource_path(':/resources/icons/gms.ico'))
    return _app_icon


def is_number(obj):
    """Returns True if the string s is a number, otherwise returns False.

        From https://stackoverflow.com/questions/354038

    Args:
        obj: Could be a string, or an int, or a float, or a tuple etc.

    Returns:
        (bool): True if the string is a number, otherwise False.
    """
    if obj is None:
        return False
    try:
        float(obj)
        return True
    except (TypeError, ValueError):
        return False


def to_float(s):
    """Converts a string to a float a little more safely (if string is empty, returns 0.0).

    Can still raise exceptions. Use is_number for better safety.

    Args:
        s (str): The string.

    Returns:
        (float): The float
    """
    return float(s) if s is not None and s.strip() else 0.0


def to_int(s):
    """Converts a string to an int a little more safely (if string is empty, returns 0).

    Can still raise exceptions. Use is_number for better safety.

    Args:
        s (str): The string.

    Returns:
        (int): The int
    """
    return int(s) if s is not None and s.strip() else 0


def get_test_files_path() -> str:
    """Returns the full path to the 'tests/files' directory.

    Returns:
        (str): See description.
    """
    files_path = os.path.join(os.path.dirname(os.path.realpath(__file__)), '..', '..', '..', 'tests', 'files')
    files_path = os.path.abspath(files_path)
    return files_path


def get_base_model_path(dis):
    """Returns the full path to the 'tests/files' directory.

    Args:
        dis(str): Must be: 'dis', 'disu', or 'disv'

    Returns:
        (str): See description.
    """
    files_path = os.path.join(os.path.dirname(os.path.realpath(__file__)), '../components/resources/base_model/', dis)
    files_path = os.path.abspath(files_path)
    return files_path


def check_all_equal(lst):
    """Returns true if all values in the list are equal.

    See https://stackoverflow.com/questions/3844801

    Args:
        lst: The list

    Returns:
        See description.
    """
    return lst is None or len(lst) == 0 or list(lst).count(lst[0]) == len(lst)


def sort_list_by_other_list(list_to_sort, list_to_sort_by):
    """Sorts a list by another list.

    See https://stackoverflow.com/questions/6618515/sorting-list-based-on-values-from-another-list

    Args:
        list_to_sort: The list to be sorted.
        list_to_sort_by: The list to sort by.

    Returns:
        The sorted version of list_to_sort.
    """
    return [x for _, x in sorted(zip(list_to_sort_by, list_to_sort), key=lambda pair: pair[0])]


def safe_cast(val, to_type, default=None):
    """Does the cast inside a try/except block.

    Args:
        val: Some value.
        to_type: Type to cast to.
        default: Default value in case cast fails.

    Returns:
        Returns either the value casted to to_type if successful, or default if not.
    """
    try:
        return to_type(val)
    except (ValueError, TypeError):
        return default


def is_non_string_sequence(obj):
    """Returns True if the object is a list, tuple, or similar, and false if it is a string or anything else.

    See https://stackoverflow.com/questions/1835018/how-to-check-if-an-object-is-a-list-or-tuple-but-not-string

    Args:
        obj: Some object.

    Returns:
        (bool): See description.
    """
    return isinstance(obj, collections.abc.Sequence) and not isinstance(obj, str)


def recursive_copy(src, dest):
    """Copy each file from src dir to dest dir, including sub-directories.

    https://stackoverflow.com/questions/3397752

    Args:
        src (str): Source directory.
        dest (str): Destination directory.
    """
    for item in os.listdir(src):
        filepath = os.path.join(src, item)

        # if item is a file, copy it
        if os.path.isfile(filepath):
            shutil.copy(filepath, dest)

        # else if item is a folder, recurse
        elif os.path.isdir(filepath):
            new_dest = os.path.join(dest, item)
            os.mkdir(new_dest)
            recursive_copy(filepath, new_dest)


def is_valid_uuid(uuid_to_test):
    """Returns True if uuid_string is a valid uuid.

    https://stackoverflow.com/questions/19989481

    Args:
        uuid_to_test (str):

    Returns:
        (bool): See description.
    """
    try:
        uuid_obj = uuid.UUID(uuid_to_test)
    except ValueError:
        return False
    return str(uuid_obj) == uuid_to_test


def filter_tree(start_node, condition):
    """Returns a tree filtered by a conditional method.

    tree_util.filter_project_explorer sucks so I wrote this.

    Args:
        start_node (TreeNode): Root of project explorer tree to filter
        condition: Method to apply to tree items. Method should take a TreeNode and
            return a bool. If True, the item and its ancestors will be included in the trimmed tree.

    Returns:
        The root of the new tree.
    """
    if not start_node:
        return None

    children = start_node.children
    if not children:  # Terminal tree item
        return None

    root = TreeNode(other=start_node)  # Create a copy of the root tree item node.
    root.children = []  # Clear children in copy of root item. Want to filter down to selectable types.
    for child in children:
        childs_tree = filter_tree(child, condition)
        if childs_tree:
            root.add_child(childs_tree)
        elif condition(child):
            root.add_child(child)

    root_matches = condition(root)
    return_root = root.children or root_matches
    return root if return_root else None


def tree_node_path(node, delimiter='/', include_project=False) -> str:
    """Returns the path to the tree node.

    Args:
        node (TreeNode):
        delimiter (str): String inserted between path nodes.
        include_project (bool): If True, includes the top level "Project" node.

    Returns:
        (str): See description.
    """
    if not node:
        return ''

    start_node = node
    path = [start_node.name]
    while start_node.parent:
        start_node = start_node.parent
        path.append(start_node.name)
    if not include_project and path[-1] == 'Project':
        del path[-1]
    path.reverse()
    return delimiter.join(path)


def find_closest_index(sorted_list: Sequence, value) -> int:
    """Returns the index of the closest value to value in a_sorted_list (which must be sorted).

    Returns the larger value if two values are equally close.

    Args:
        sorted_list: A sorted list of values.
        value: A value (not necessarily in the list)

    Returns:
        See description.
    """
    i = bisect.bisect_left(sorted_list, value)
    if i >= len(sorted_list):
        i = len(sorted_list) - 1
    elif i and sorted_list[i] - value > value - sorted_list[i - 1]:
        i = i - 1
    return i


def mean(a_list):
    """Returns the mean of the items in the list."""
    if not a_list:
        return 0.0
    return statistics.mean(a_list)


def mae(a_list):
    """Returns the mean absolute error of the items in the list."""
    if not a_list:
        return 0.0
    return np.sum(np.absolute(a_list)) / len(a_list)


def rms(a_list):
    """Returns the root mean squared error of the items in the list."""
    if not a_list:
        return 0.0
    ms = 0.0
    for item in a_list:
        ms += pow(item, 2.0)
    return math.sqrt(ms / len(a_list))


def file_hack(filename: str) -> bool:
    """Returns True if the file exists in C:/temp.

    Args:
        filename:

    Returns:
        See description.
    """
    return Path(f'C:/temp/{filename}').is_file()


def clamp(n, smallest, largest):
    """Return n, adjusted, if necessary, between smallest and largest.

    Args:
        n:
        smallest:
        largest:

    Returns:
        See description.
    """
    return max(smallest, min(n, largest))


def null_path(filepath: Pathlike) -> bool:
    """Returns True if the path is 'null'; '' if a str, or '.' if a pathlib.Path."""
    if filepath is None:
        return True
    elif isinstance(filepath, str) and filepath == '':
        return True
    elif isinstance(filepath, Path) and str(filepath) == '.':
        return True
    else:
        return False


def merge_sequences(seq1, seq2):
    """Merge two sequences using difflib to preserve relative order.

    See https://stackoverflow.com/a/14242169.

    Args:
        seq1: First sequence.
        seq2: Second sequence.

    Returns:
        The merged sequence.
    """
    sm = SequenceMatcher(a=seq1, b=seq2)
    res = []
    for (op, start1, end1, start2, end2) in sm.get_opcodes():
        if op == 'equal' or op == 'delete':
            # This range appears in both sequences, or only in the first one.
            res += seq1[start1:end1]
        elif op == 'insert':
            # This range appears in only the second sequence.
            res += seq2[start2:end2]
        elif op == 'replace':
            # There are different ranges in each sequence - add both.
            res += seq1[start1:end1]
            res += seq2[start2:end2]
    return res
