"""Module for ExportSimThread class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['ExportSimThread']

# 1. Standard Python modules
from datetime import timedelta
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.dmi.xms_data import XmsData
from xms.constraint import GridType as CoGridType
from xms.gmi.data.generic_model import Section
from xms.gmi.data_bases.sim_base_data import SimBaseData
from xms.grid.ugrid import UGrid
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.feedback_thread import ExitError, FeedbackThread
from xms.guipy.time_format import datetime_to_string
from xms.guipy.widgets.widget_builder import datetime_from_string

# 4. Local modules
from xms.ptm.components.sources_component import PtmSourcesComponent
from xms.ptm.components.traps_component import PtmTrapsComponent
from xms.ptm.file_io.control import write_control
from xms.ptm.file_io.datasets import write_datasets
from xms.ptm.file_io.grid import write_grid
from xms.ptm.file_io.sources.source_writer import write_sources
from xms.ptm.file_io.traps.trap_writer import write_traps
from xms.ptm.model.model import HydroDefinition, MeshDefinition, SedimentDefinition, simulation_model, StopMode


class ExportSimThread(FeedbackThread):
    """Import thread."""
    def __init__(self, data: XmsData):
        """
        Construct the worker.

        Args:
            data: Interprocess communication object.
        """
        super().__init__(xms_data=data)
        self.display_text |= {
            'title': 'Reading PTM Control File',
            'working_prompt': 'Reading control file. Please wait...',
        }
        self._data = data
        self._ok = True
        self._stop_time: str = ''
        self._model_control: Optional[Section] = None

    def _run(self):
        """Run the thread."""
        self._log.info('Exporting simulation...')

        self._get_model_control()

        self._check_grid()
        self._check_sources()
        self._check_traps()
        self._check_hydrodynamic_datasets()
        self._check_sediments()
        self._check_neighbors_and_bc()

        if not self._ok:
            raise ExitError()

        self._write_grid()
        self._write_sources()
        self._write_traps()
        self._write_hydrodynamics()
        self._write_sediments()
        self._write_neighbors_and_bc()
        self._write_control()  # Must be last since other steps might alter file paths.

    def _get_model_control(self):
        """Initialize self._model_control."""
        sim_data: SimBaseData = self._data.simulation_data
        section = simulation_model()
        section.restore_values(sim_data.global_values)
        self._model_control = section

        time = self._model_control.group('time')
        if time.parameter('stop_mode').value == StopMode.date_time:
            self._stop_time = time.parameter('stop_run').value
        else:
            start = datetime_from_string(time.parameter('start_run').value)
            seconds = float(time.parameter('duration').value)
            duration = timedelta(seconds=seconds)
            end = start + duration
            self._stop_time = datetime_to_string(end)

    def _check_grid(self):
        """Check that a linked or external grid is valid."""
        mesh = self._model_control.group('mesh')
        mesh_type = mesh.parameter('mesh_type')

        if mesh_type.value == MeshDefinition.linked:
            self._check_linked_grid()
        else:
            self._check_external_grid()

    def _check_linked_grid(self):
        """Check that a linked grid is valid."""
        grid = self._data.linked_grid
        if not grid:
            self._ok = False
            self._log.error('Simulation is set to use linked grid, but no mesh or grid was linked to the simulation.')
            return

        if grid.grid_type == CoGridType.quadtree_2d or grid.grid_type == CoGridType.rectilinear_2d:
            self._ok = False
            self._log.error(
                'The PTM interface only supports triangle cells. Quadtrees and rectilinear grids can be triangulated '
                'by using the "2D Mesh from 2D UGrid" tool on the grid\'s cell centers. After triangulation, existing '
                'datasets can be interpolated to the new mesh from the right-click menu on the original grid.'
            )
            return

        if not grid.check_all_cells_are_of_type(UGrid.cell_type_enum.TRIANGLE):
            self._ok = False
            self._log.error(
                'The PTM interface only supports triangle cells. Non-triangle cells must be converted to '
                'triangles before exporting.'
            )

    def _check_external_grid(self):
        """Check that an external grid is valid."""
        external_file = self._model_control.group('mesh').parameter('mesh_file').value

        if not external_file:
            self._ok = False
            self._log.error('Simulation is set to use external grid, but no external grid file was set.')
            return

        if not Path(external_file).exists():
            self._ok = False
            self._log.error('Simulation is set to use external grid, but external grid file does not exist.')
            return

    def _write_grid(self):
        """Write the grid file."""
        mesh = self._model_control.group('mesh')
        mesh_type = mesh.parameter('mesh_type').value
        if mesh_type != MeshDefinition.linked:
            return

        mesh.parameter('mesh_file').value = 'mesh.grd'
        write_grid(self._data.linked_grid, 'mesh.grd')

    def _check_neighbors_and_bc(self):
        """Check that the .neighbors and .bc file parameters are valid."""
        mesh = self._model_control.group('mesh')
        mesh_type = mesh.parameter('mesh_type').value
        if mesh_type == MeshDefinition.linked:
            return

        neighbor_file = mesh.parameter('neighbor_file').value
        if not neighbor_file:
            self._ok = False
            self._log.error('External .bc and .neighbor files are in use, but .neighbors file path was not set.')
        if not Path(neighbor_file).exists():
            message = (
                'External .bc and .neighbor files are in use, but .neighbors file path refers to nonexistent file. '
                'File will be generated when simulation is run.'
            )
            self._log.warning(message)

        bc_file = mesh.parameter('bc_file').value
        if not bc_file:
            self._ok = False
            self._log.error('External .bc and .neighbor files are in use, but .bc file path was not set.')
        if not Path(bc_file).exists():
            message = (
                'External .bc and .neighbor files are in use, but .bc file path refers to nonexistent file. '
                'File will be generated when simulation is run.'
            )
            self._log.warning(message)

    def _write_neighbors_and_bc(self):
        """
        Write the .neighbors and .bc files.

        There isn't actually anything to write, this really just fixes the output names as necessary.
        """
        mesh = self._model_control.group('mesh')
        mesh_type = mesh.parameter('mesh_type').value
        if mesh_type != MeshDefinition.linked:
            return

        mesh.parameter('bc_file').value = 'mesh.bc'
        mesh.parameter('neighbor_file').value = 'mesh.neighbors'

    def _write_control(self):
        """Write the control file."""
        self._log.info('Writing control file...')

        simulation_name = self._data.simulation_name
        write_control(self._model_control, f'{simulation_name}.pcf')

    def _check_sources(self):
        """Check that the source coverage can be written and log warnings if not."""
        coverage, data = self._data.get_linked_coverage(PtmSourcesComponent)
        if not coverage:
            self._ok = False
            self._log.error('Exporting requires a PTM Sources coverage linked to the simulation.')
            return

        have_points = data.component_id_map[TargetType.point]
        have_arcs = data.component_id_map[TargetType.arc]
        have_polygons = data.component_id_map[TargetType.polygon]

        if not have_points and not have_arcs and not have_polygons:
            self._ok = False
            self._log.error('The linked Sources coverage has no assigned features.')
            return

        where = self._model_control.group('files').parameter('source_file').value
        if not where:
            self._ok = False
            self._log.error('No source file name specified in the model control.')
            return

    def _write_sources(self):
        """Write the sources."""
        where = self._model_control.group('files').parameter('source_file').value

        coverage, data = self._data.get_linked_coverage(PtmSourcesComponent)
        write_sources(coverage, data, self._stop_time, where)

    def _check_traps(self):
        """
        Check that the trap coverage is okay and log warnings if not.

        Sets self._ok=False if an unrecoverable error is encountered.
        """
        coverage, data = self._data.get_linked_coverage(PtmTrapsComponent)
        if not coverage:
            return

        where = self._model_control.group('files').parameter('trap_file').value
        if not where:
            self._ok = False
            self._log.error('No trap file name specified in the model control.')
            return

    def _write_traps(self):
        """Write the traps if necessary."""
        coverage, data = self._data.get_linked_coverage(PtmTrapsComponent)
        if not coverage:
            return

        where = self._model_control.group('files').parameter('trap_file').value

        coverage, data = self._data.get_linked_coverage(PtmTrapsComponent)
        write_traps(coverage, data, where)

    def _check_hydrodynamic_datasets(self) -> None:
        """Check the hydrodynamic datasets."""
        mesh = self._model_control.group('mesh')
        hydro_definition = mesh.parameter('hydro_type').value

        if hydro_definition == HydroDefinition.linked:
            self._check_internal_hydro_datasets()
        else:
            self._check_external_hydro_datasets()

    def _check_external_hydro_datasets(self):
        """Check that external hydro datasets are valid."""
        mesh = self._model_control.group('mesh')
        flow_file = mesh.parameter('xmdf_hydro_file').value
        if not flow_file:
            self._ok = False
            self._log.error('Flow file was not selected in the model control.')
        if not Path(flow_file).exists():
            self._ok = False
            self._log.warning(f'External flow file does not exist yet: {flow_file}')

        elevation_path = mesh.parameter('xmdf_elevation_path').value
        if not elevation_path:
            self._ok = False
            self._log.error('Elevation dataset path was not selected in the model control.')
        flow_path = mesh.parameter('xmdf_flow_path').value
        if not flow_path:
            self._ok = False
            self._log.error('Flow dataset path was not selected in the model control.')

    def _check_internal_hydro_datasets(self):
        """Check that internal hydro datasets are valid."""
        if not self._data.linked_grid:
            # Linked datasets require a linked UGrid. If we're here, the UGrid checks should log an error for us.
            self._ok = False
            return

        mesh = self._model_control.group('mesh')

        mesh_definition = mesh.parameter('mesh_type').value
        if mesh_definition != MeshDefinition.linked:
            self._ok = False
            self._log.error('Cannot use linked hydro datasets with an external mesh/grid.')

        elevation_uuid = mesh.parameter('linked_elevation_dataset').value
        if not elevation_uuid:
            self._ok = False
            self._log.error('Elevation dataset was not selected in the model control.')
        elif not self._data.datasets[elevation_uuid]:
            self._ok = False
            self._log.error('Elevation dataset selected in the model control no longer exists.')
        elif self._data.datasets[elevation_uuid].geom_uuid != self._data.linked_grid.uuid:
            self._ok = False
            self._log.error('"Elevation Dataset" set in model control does not match linked mesh.')

        flow_uuid = mesh.parameter('linked_flow_dataset').value
        if not flow_uuid:
            self._ok = False
            self._log.error('Flow dataset was not selected in the model control.')
        elif not self._data.datasets[flow_uuid]:
            self._ok = False
            self._log.error('Flow dataset selected in the model control no longer exists.')
        elif self._data.datasets[flow_uuid].geom_uuid != self._data.linked_grid.uuid:
            self._ok = False
            self._log.error('"Flow Dataset" set in model control does not match linked mesh.')

    def _write_hydrodynamics(self):
        """Write the hydrodynamic datasets."""
        mesh = self._model_control.group('mesh')
        if mesh.parameter('hydro_type').value != HydroDefinition.linked:
            return

        mesh.parameter('hydro_type').value = HydroDefinition.adcirc_xmdf
        mesh.parameter('xmdf_hydro_file').value = 'hydro.h5'
        mesh.parameter('xmdf_elevation_path').value = 'Datasets/elevation'
        mesh.parameter('xmdf_flow_path').value = 'Datasets/flow'

        flow_file = mesh.parameter('xmdf_hydro_file').value

        elevation_name = 'elevation'
        elevation_uuid = mesh.parameter('linked_elevation_dataset').value
        elevation_dataset = self._data.datasets[elevation_uuid]

        flow_name = 'flow'
        flow_uuid = mesh.parameter('linked_flow_dataset').value
        flow_dataset = self._data.datasets[flow_uuid]

        datasets = [(elevation_dataset, elevation_name), (flow_dataset, flow_name)]
        write_datasets(datasets, flow_file)

    def _check_sediments(self) -> None:
        """Check the sediment datasets."""
        mesh = self._model_control.group('mesh')
        sediment_definition = mesh.parameter('sediment_type').value

        if sediment_definition == SedimentDefinition.linked:
            self._check_internal_sediment_datasets()
        else:
            self._check_external_sediment_datasets()

    def _check_external_sediment_datasets(self):
        """Check that external sediment datasets are valid."""
        mesh = self._model_control.group('mesh')
        sediment_definition = mesh.parameter('sediment_type').value

        sediment_file = mesh.parameter('sediment_file').value
        if not sediment_file:
            self._ok = False
            self._log.error('Sediment file was not selected in the model control.')
        if not Path(sediment_file).exists():
            self._ok = False
            self._log.warning(f'External sediment file does not exist yet: {sediment_file}')

        d35_path = mesh.parameter('d35_path').value
        if sediment_definition == SedimentDefinition.xmdf and not d35_path:
            self._ok = False
            self._log.error('"D35 Path" value not set in the model control.')
        d50_path = mesh.parameter('d50_path').value
        if sediment_definition == SedimentDefinition.xmdf and not d50_path:
            self._ok = False
            self._log.error('"D50 Path" value not set in the model control.')
        d90_path = mesh.parameter('d90_path').value
        if sediment_definition == SedimentDefinition.xmdf and not d90_path:
            self._ok = False
            self._log.error('"D90 Path" value not set in the model control.')

    def _check_internal_sediment_datasets(self):
        """Check that internal sediment datasets are valid."""
        if not self._data.linked_grid:
            # Linked datasets require a linked UGrid. If we're here, the UGrid checks should log an error for us.
            self._ok = False
            return

        mesh = self._model_control.group('mesh')

        mesh_definition = mesh.parameter('mesh_type').value
        if mesh_definition != MeshDefinition.linked:
            self._ok = False
            self._log.error('Cannot use linked sediment datasets with an external mesh/grid.')

        d35_uuid = mesh.parameter('d35_dataset').value
        if not d35_uuid:
            self._ok = False
            self._log.error('"D35 Dataset" value not set in the model control.')
        elif not self._data.datasets[d35_uuid]:
            self._ok = False
            self._log.error('"D35 Dataset" set in model control no longer exists.')
        elif self._data.datasets[d35_uuid].geom_uuid != self._data.linked_grid.uuid:
            self._ok = False
            self._log.error('"D35 Dataset" set in model control does not match linked mesh.')

        d50_uuid = mesh.parameter('d50_dataset').value
        if not d50_uuid:
            self._ok = False
            self._log.error('"D50 Dataset" value not set in the model control.')
        elif not self._data.datasets[d50_uuid]:
            self._ok = False
            self._log.error('"D50 Dataset" set in model control no longer exists.')
        elif self._data.datasets[d50_uuid].geom_uuid != self._data.linked_grid.uuid:
            self._ok = False
            self._log.error('"D50 Dataset" set in model control does not match linked mesh.')

        d90_uuid = mesh.parameter('d90_dataset').value
        if not d90_uuid:
            self._ok = False
            self._log.error('"D90 Dataset" value not set in the model control.')
        elif not self._data.datasets[d90_uuid]:
            self._ok = False
            self._log.error('"D90 Dataset" set in model control no longer exists.')
        elif self._data.datasets[d90_uuid].geom_uuid != self._data.linked_grid.uuid:
            self._ok = False
            self._log.error('"D90 Dataset" set in model control does not match linked mesh.')

    def _write_sediments(self):
        """Write the sediment datasets."""
        mesh = self._model_control.group('mesh')
        if mesh.parameter('sediment_type').value != SedimentDefinition.linked:
            return

        mesh.parameter('sediment_type').value = SedimentDefinition.xmdf
        mesh.parameter('sediment_file').value = 'sediments.h5'

        d35_name = 'd35'
        d35_uuid = mesh.parameter('d35_dataset').value
        d35_dataset = self._data.datasets[d35_uuid]
        mesh.parameter('d35_path').value = 'Datasets/d35'

        d50_name = 'd50'
        d50_uuid = mesh.parameter('d50_dataset').value
        d50_dataset = self._data.datasets[d50_uuid]
        mesh.parameter('d50_path').value = 'Datasets/d50'

        d90_name = 'd90'
        d90_uuid = mesh.parameter('d90_dataset').value
        d90_dataset = self._data.datasets[d90_uuid]
        mesh.parameter('d90_path').value = 'Datasets/d90'

        datasets = [(d35_dataset, d35_name), (d50_dataset, d50_name), (d90_dataset, d90_name)]
        write_datasets(datasets, 'sediments.h5')
