"""Module for ImportDatasetsThread class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['ImportDatasetsThread']

# 1. Standard Python modules
from datetime import datetime
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.adcirc.file_io.solution_importer import ADCIRCSolutionReader
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.dmi.xms_data import XmsData
from xms.datasets.dataset_reader import DatasetReader
from xms.gmi.data.generic_model import Section
from xms.guipy.dialogs.feedback_thread import ExpectedError, FeedbackThread

# 4. Local modules
from xms.ptm.model.model import HydroDefinition, simulation_model


class ImportDatasetsThread(FeedbackThread):
    """Import thread."""
    def __init__(self, data: XmsData):
        """
        Construct the worker.

        Args:
            data: Interprocess communication object.
        """
        super().__init__(xms_data=data)
        self.display_text |= {
            'title': 'Reading dataset files...',
            'working_prompt': 'Reading dataset files. Please wait...',
        }
        self._data = data
        self._control: Section = simulation_model()

        self._elevation_dataset: Optional[DatasetReader] = None
        self._flow_dataset: Optional[DatasetReader] = None

    def _run(self):
        """Run the thread."""
        self._get_control()
        self._read_flow()
        self._read_sediments()

        if self._elevation_dataset and self._flow_dataset:
            self._data.add_dataset(self._elevation_dataset)
            self._data.add_dataset(self._flow_dataset)

    def _read_flow(self):
        """Read the flow datasets in whatever format they happen to be in."""
        flow_format = self._control.group('mesh').parameter('hydro_type').value
        if flow_format == HydroDefinition.adcirc_ascii:
            self._read_flow_ascii_adcirc()
        elif flow_format in [HydroDefinition.cms_multi, HydroDefinition.adcirc_xmdf]:
            self._read_flow_multi_xmdf()
        elif flow_format == HydroDefinition.cms_single:
            self._read_flow_single_xmdf()
        else:
            raise AssertionError('Unsupported flow format.')

    def _read_flow_ascii_adcirc(self):
        """Read the flow datasets in ADCIRC ASCII format."""
        ok = True
        elevation_file: str = self._control.group('mesh').parameter('adcirc_ascii_elevation_file').value
        flow_file: str = self._control.group('mesh').parameter('adcirc_ascii_flow_file').value

        if not elevation_file:
            self._log.error('ADCIRC ASCII elevation file not set in model control.')
            ok = False
        elif not Path(elevation_file).exists():
            self._log.error('ADCIRC ASCII elevation file set in the model control does not exist.')
            ok = False

        if not flow_file:
            self._log.error('ADCIRC ASCII flow file not set in model control.')
            ok = False
        elif not Path(flow_file).exists():
            self._log.error('ADCIRC ASCII flow file set in the model control does not exist.')
            ok = False

        if not ok:
            return

        reftime = datetime(year=1, month=1, day=1)
        temp_dir = XmEnv.xms_environ_process_temp_directory()

        reader = ADCIRCSolutionReader([elevation_file], reftime, temp_dir)
        reader.read()
        if not reader.datasets:
            # The ADCIRC reader already logged an error for us.
            return
        elevation_dataset = reader.datasets[0]

        reader = ADCIRCSolutionReader([flow_file], reftime, temp_dir)
        reader.read()
        if not reader.datasets:
            # The ADCIRC reader fails silently in this case.
            self._log.error(f'Error reading solution file: {flow_file}')
            return
        flow_dataset = reader.datasets[0]

        for dataset in [elevation_dataset, flow_dataset]:
            reader = DatasetReader(dataset.filename, group_path=dataset.group_path)
            self._data.add_dataset(reader)

    def _read_flow_single_xmdf(self):
        """Read the flow datasets in single-XMDF format."""
        ok = True
        hydro_file: str = self._control.group('mesh').parameter('xmdf_hydro_file').value
        elevation_path: str = self._control.group('mesh').parameter('xmdf_elevation_path').value
        flow_path: str = self._control.group('mesh').parameter('xmdf_flow_path').value

        if not hydro_file:
            self._log.error('XMDF hydro file not set in model control.')
            ok = False
        elif not Path(hydro_file).exists():
            self._log.error('XMDF hydro file set in the model control does not exist.')
            ok = False

        if not elevation_path:
            self._log.error('Elevation dataset path not set in model control.')
            ok = False
        if not flow_path:
            self._log.error('Flow dataset path not set in model control.')
            ok = False

        if not ok:
            return

        try:
            self._elevation_dataset = DatasetReader(str(hydro_file), group_path=elevation_path)
        except KeyError:
            raise ExpectedError('Unable to read elevation dataset.')
        try:
            self._flow_dataset = DatasetReader(str(hydro_file), group_path=flow_path)
        except KeyError:
            raise ExpectedError('Unable to read flow dataset.')

    def _read_flow_multi_xmdf(self):
        """Read the flow datasets in multi-XMDF format."""
        ok = True
        elevation_file: str = self._control.group('mesh').parameter('xmdf_elevation_file').value
        elevation_path: str = self._control.group('mesh').parameter('xmdf_elevation_path').value
        flow_file: str = self._control.group('mesh').parameter('xmdf_flow_file').value
        flow_path: str = self._control.group('mesh').parameter('xmdf_flow_path').value

        if not elevation_file:
            self._log.error('Elevation XMDF file not set in model control.')
            ok = False
        elif not Path(elevation_file).exists():
            self._log.error('Elevation XMDF file set in the model control does not exist.')
            ok = False
        if not elevation_path:
            self._log.error('Elevation dataset path not set in model control.')
            ok = False

        if not flow_file:
            self._log.error('Flow XMDF file not set in model control.')
            ok = False
        elif not Path(flow_file).exists():
            self._log.error('Flow XMDF file set in the model control does not exist.')
            ok = False
        if not flow_path:
            self._log.error('Flow dataset path not set in model control.')
            ok = False

        if not ok:
            return

        try:
            self._elevation_dataset = DatasetReader(str(elevation_file), group_path=elevation_path)
        except KeyError:
            raise ExpectedError('Unable to read elevation dataset.')
        try:
            self._flow_dataset = DatasetReader(str(flow_file), group_path=flow_path)
        except KeyError:
            raise ExpectedError('Unable to read flow dataset.')

    def _read_sediments(self):
        """Read the sediment datasets."""
        return
        # TODO
        # sediment_format = self._control.group('mesh').parameter('sediment_format').value
        # if sediment_format == SedimentType.adcirc:
        #     raise NotImplementedError()
        # elif sediment_format == SedimentType.xmdf:
        #     raise NotImplementedError()
        # elif sediment_format == SedimentType.m2d:
        #     raise NotImplementedError()
        # elif sediment_format == SedimentType.uniform:
        #     raise NotImplementedError()
        # else:
        #     raise AssertionError('Unsupported sediment type.')

    def _get_control(self):
        """Get the model control."""
        self._log.info('Retrieving model control...')
        sim_data = self._data.simulation_data
        self._control.restore_values(sim_data.global_values)
