"""Module for the trap writer."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['write_traps']

# 1. Standard Python modules
from pathlib import Path
from typing import TextIO

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.ptmio.trap.traps import Direction, LineTrap, PolygonTrap, Traps


def write_traps(traps: Traps, where: str | Path | TextIO):
    """
    Write traps to a file.

    Args:
        traps: The traps to write.
        where: Where to write them.
    """
    if isinstance(where, (str, Path)):
        with open(where, 'w') as f:
            write_traps(traps, f)
            return

    _write_line_traps(traps.line_traps, where)
    _write_polygon_traps(traps.polygon_traps, where)


def _write_line_traps(traps: list[LineTrap], where: TextIO):
    """
    Write all the line traps.

    Args:
        traps: The traps to write.
        where: Where to write them.
    """
    where.write(f'{len(traps)} ! Number of line traps\n')

    for trap in traps:
        _write_line_trap(trap, where)


def _write_line_trap(trap: LineTrap, where: TextIO):
    """
    Write a single line trap.

    Args:
        trap: The trap to write.
        where: Where to write it.
    """
    if trap.direction == Direction.decreasing_x:
        direction = -1
    elif trap.direction == Direction.any:
        direction = 0
    elif trap.direction == Direction.increasing_x:
        direction = 1
    else:
        raise AssertionError('Unknown direction')

    open_flag = 1 if trap.is_open else 0
    single_flag = 1 if trap.is_single else 0

    where.write(f'{trap.trap_id} {trap.label}\n')
    where.write(f' {direction} {trap.bottom} {trap.top} {open_flag} {single_flag}\n')
    where.write(f' {trap.start[0]} {trap.start[1]}\n')
    where.write(f' {trap.end[0]} {trap.end[1]}\n')


def _write_polygon_traps(traps: list[PolygonTrap], where: TextIO):
    """
    Write all the polygon traps to the file.

    Args:
        traps: Polygon traps to write.
        where: Where to write them.
    """
    where.write(f'{len(traps)} ! Number of polygon traps\n')

    for trap in traps:
        _write_polygon_trap(trap, where)


def _write_polygon_trap(trap: PolygonTrap, where: TextIO):
    """
    Write a single polygon trap.

    Args:
        trap: The trap to write.
        where: Where to write it.
    """
    open_flag = 1 if trap.is_open else 0
    single_flag = 1 if trap.is_single else 0
    num_points = len(trap.points)

    where.write(f'{trap.trap_id} {trap.label}\n')
    where.write(f' {num_points} {trap.bottom} {trap.top} {open_flag} {single_flag}\n')

    for number, (x, y) in enumerate(trap.points, start=1):
        where.write(f' {number} {x} {y}\n')
