"""RSM control file writer."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from xml.dom import minidom
import xml.etree.cElementTree as Et

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Section
from xms.guipy.time_format import string_to_datetime

# 4. Local modules
from xms.rsm.data import sim_data_def as sdd
from xms.rsm.file_io import util
from xms.rsm.file_io.basin_writer import BasinWriter
from xms.rsm.file_io.canal_writer import CanalWriter
from xms.rsm.file_io.cell_monitor_writer import CellMonitorWriter
from xms.rsm.file_io.csv_writer import CsvWriter
from xms.rsm.file_io.impoundment_writer import ImpoundmentWriter
from xms.rsm.file_io.lake_writer import LakeWriter
from xms.rsm.file_io.mesh_bc_writer import MeshBcWriter
from xms.rsm.file_io.mesh_data_writer import MeshDataWriter
from xms.rsm.file_io.rule_curve_writer import RuleCurveWriter
from xms.rsm.file_io.water_mover_writer import WaterMoverWriter
from xms.rsm.file_io.wcd_writer import WcdWriter
from xms.rsm.file_io.writer_data import WriterData


class ControlWriter:
    """Writer class for the RSM control file."""
    def __init__(self, xms_data):
        """Constructor.

        Args:
            xms_data (XmsData): Simulation data retrieved from SMS
        """
        self._filename = ''
        self._logger = util.get_logger()
        self._xms_data = xms_data
        self._mesh_file_name = ''
        self._output_xml = None
        self._csv_writer = CsvWriter()
        self._writer_data = WriterData(self._xms_data, self._csv_writer)
        self._mesh_ds_writer = None
        self._basin_writer = None
        self._canal_writer = None
        self._impoundment_writer = None
        self._lake_writer = None
        self._rule_curve_writer = None
        self._wcd_writer = None
        self._water_mover_writer = None
        self._sim_global_parameters = None

    def write(self):
        """Write the RSM control file."""
        try:
            self._export_2dm()
            self._export_rsm_xml()
        except Exception as e:
            self._logger.error(f'Error writing RSM control file: {e}')

    def _get_sim_global_parameters(self):
        """Get the simulation global parameters.

        Returns:
            dict: The simulation global parameters.
        """
        gm = sdd.generic_model()
        gm.global_parameters.restore_values(self._xms_data.sim_component.data.global_values)
        self._xms_data.sim_global_parameters = gm.global_parameters

    def _export_rsm_xml(self):
        """Write the RSM xml file."""
        self._logger.info('Writing RSM xml input file.')
        self._writer_data.xml_hse = Et.Element('hse')
        self._get_sim_global_parameters()
        atts = _simulation_control_data_for_export(self._xms_data.sim_global_parameters)
        Et.SubElement(self._writer_data.xml_hse, 'control', atts)
        self._write_rule_curves()

        self._writer_data.xml_mesh = Et.SubElement(self._writer_data.xml_hse, 'mesh')
        Et.SubElement(self._writer_data.xml_mesh, 'geometry', {'file': self._mesh_file_name})
        self._add_mesh_properties()
        self._add_mesh_bcs()
        self._add_network()
        self._add_lakes()
        self._add_impoundments()
        self._add_basins()
        self._add_wcds()
        self._add_water_movers()
        self._add_hpm()  # must do this after water bodies because hpm may reference a water body

        self._add_outputs()

        xmlstr = minidom.parseString(Et.tostring(self._writer_data.xml_hse)).toprettyxml(indent='  ')
        xmlstr = xmlstr.replace('<remove_me>', '')
        xmlstr = xmlstr.replace('</remove_me>', '')
        xml_fname = f'{self._xms_data.sim_name.replace(" ", "_")}.xml'
        with open(xml_fname, 'w') as f:
            f.write(xmlstr)

    def _write_rule_curves(self):
        """Write the rule curves to the XML."""
        writer = RuleCurveWriter(self._writer_data)
        writer.write()

    def _add_mesh_properties(self):
        """Add properties to the mesh portion of the XML."""
        self._mesh_ds_writer = MeshDataWriter(self._writer_data)
        self._mesh_ds_writer.write()

    def _add_hpm(self):
        """Add the hpm portion of the XML."""
        self._mesh_ds_writer.write_hpms()

    def _add_mesh_bcs(self):
        """Add properties to the mesh portion of the XML."""
        mesh_bc_writer = MeshBcWriter(self._writer_data)
        mesh_bc_writer.write()

    def _add_network(self):
        """Add the network portion of the XML."""
        self._canal_writer = CanalWriter(self._writer_data)
        self._canal_writer.write()

    def _add_lakes(self):
        """Add the lakes portion of the XML."""
        self._lake_writer = LakeWriter(self._writer_data)
        self._lake_writer.write()

    def _add_impoundments(self):
        """Add the lakes portion of the XML."""
        self._impoundment_writer = ImpoundmentWriter(self._writer_data)
        self._impoundment_writer.write()

    def _add_basins(self):
        """Add the lakes portion of the XML."""
        self._basin_writer = BasinWriter(self._writer_data)
        self._basin_writer.write()

    def _add_wcds(self):
        """Add the water control districts portion of the XML."""
        self._wcd_writer = WcdWriter(self._writer_data)
        self._wcd_writer.write()

    def _add_water_movers(self):
        """Add the watermovers portion of the XML."""
        self._water_mover_writer = WaterMoverWriter(self._writer_data)
        self._water_mover_writer.write()
        self._canal_writer.append_bcs(self._water_mover_writer.canal_bcs)

    def _export_2dm(self):
        """Get the mesh associated with the simulation and write to a 2dm file."""
        ug = self._xms_data.xmugrid
        if ug is None:
            raise ValueError('No UGrid in the simulation. Aborting.')
        self._mesh_file_name = self._xms_data.do_ugrid.name.replace(' ', '_') + '.2dm'
        with open(self._mesh_file_name, 'w') as f:
            f.write('MESH2D\n')
            cs = ug.cellstream
            cnt = 0
            cell_id = 0
            while cnt < len(cs):
                cell_id += 1
                # cell_type = cs[cnt]
                num_pts = cs[cnt + 1]
                start = cnt + 2
                end = start + num_pts
                cell_pts = cs[start:end]
                elem = 'E3T' if num_pts == 3 else 'E4Q'
                f.write(f'{elem} {cell_id} {cell_pts[0] + 1} {cell_pts[1] + 1} {cell_pts[2] + 1}')
                if num_pts > 3:
                    f.write(f' {cell_pts[3] + 1}')
                f.write('\n')
                cnt = end
            locs = ug.locations
            for idx, loc in enumerate(locs):
                f.write(f'ND {idx + 1} {loc[0]} {loc[1]} {loc[2]}\n')

    def _add_outputs(self):
        """Add the output portion of the XML."""
        self._output_xml = Et.SubElement(self._writer_data.xml_hse, 'output')
        self._global_outputs()
        cell_writer = CellMonitorWriter(self._output_xml, self._xms_data)
        cell_writer.write()
        self._canal_writer.write_monitor(self._output_xml)
        self._lake_writer.write_monitor(self._output_xml)
        self._water_mover_writer.write_monitor(self._output_xml)
        self._impoundment_writer.write_monitor(self._output_xml)
        self._basin_writer.write_monitor(self._output_xml)
        self._wcd_writer.write_monitor(self._output_xml)
        if len(self._output_xml) < 1:
            self._writer_data.xml_hse.remove(self._output_xml)

    def _global_outputs(self):
        """Add the global outputs to the XML."""
        gp = self._xms_data.sim_global_parameters
        grp = gp.group('output')
        if grp.parameter('wb_budget').value:
            atts = {'file': grp.parameter('wb_budget_filename').value}
            if grp.parameter('wb_dbintl').value > 0:
                atts['dbintl'] = f'{grp.parameter("wb_dbintl").value}'
            Et.SubElement(self._output_xml, 'wbbudgetpackage', attrib=atts)
        if grp.parameter('hpm_budget').value:
            atts = {'file': grp.parameter('hpm_budget_filename').value}
            if grp.parameter('hpm_dbintl').value > 0:
                atts['dbintl'] = f'{grp.parameter("hpm_dbintl").value}'
            Et.SubElement(self._output_xml, 'hpmbudgetpackage', attrib=atts)


def _simulation_control_data_for_export(global_parameters: Section) -> dict:
    """Get the simulation control data for export.

    Args:
        sim_component: The simulation component.

    Returns:
        (str): The simulation control text for export.
    """
    gp = global_parameters
    control = gp.group('control')
    start_str = control.parameter('start_date').value
    start_date = string_to_datetime(start_str).strftime('%d%b%Y').lower()
    end_str = control.parameter('end_date').value
    end_date = string_to_datetime(end_str).strftime('%d%b%Y').lower()
    units = 'english'
    xml_data = {
        'tslen': f'{control.parameter("timestep_length").value}',
        'tstype': control.parameter('timestep_type').value,
        'startdate': start_date,
        'starttime': f'{control.parameter("start_time").value:04d}',
        'enddate': end_date,
        'endtime': f'{control.parameter("end_time").value:04d}',
        'alpha': f'{control.parameter("timestep_weighting").value}',
        'solver': 'PETSC',
        'method': control.parameter('solver_method').value,
        'precond': control.parameter('solver_preconditioner').value,
        'units': units,
        'runDescriptor': control.parameter('run_descriptor').value,
    }
    return xml_data
