"""Module for CoverageBuilder class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['CoverageComponentBuilder', 'UNASSIGNED_TYPE', 'MULTIPLE_TYPES']

# 1. Standard Python modules
from itertools import count
from pathlib import Path
from typing import Any, Optional, Sequence
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_helper import DisplayOptionsHelper, MULTIPLE_TYPES, UNASSIGNED_TYPE
from xms.constraint import UGrid2d
from xms.data_objects.parameters import Arc, Coverage, Point, Polygon
from xms.guipy.data.target_type import TargetType

# 4. Local modules


class CoverageComponentBuilder:
    """Class for building a coverage using node IDs in a UGrid rather than locations."""
    def __init__(self, main_file: str | Path, name: str, ugrid: Optional[UGrid2d] = None):
        """
        Initialize the builder.

        Args:
            main_file: The main file of the component whose coverage is being built. Used to initialize display options.
            name: Name to assign the coverage. Appears in the UI.
            ugrid: Reference geometry for building features from.
        """
        self._main_file = main_file
        self._locations: Optional[Sequence[int]] = ugrid.ugrid.locations if ugrid is not None else None

        self._used_point_indexes: set[int] = set()
        self._used_point_feature_ids: set[int] = set()
        self._points: list[Point] = []
        self._point_component_ids: list[int] = []

        self._used_arc_ids: set[int] = set()
        self._arcs = []
        self._arc_component_ids: list[int] = []

        self._polygons: list[Polygon] = []
        self._used_polygon_ids: set[int] = set()
        self._polygon_component_ids: list[int] = []

        self._coverage = Coverage(name=name, uuid=str(uuid.uuid4()))

    def _get_point(self, node_index: int, feature_id: int) -> Point:
        """
        Get a point.

        If the same point is requested multiple times, the same one will be returned.

        Args:
            node_index: Index of a node in the mesh to get a point for.
            feature_id: ID to assign the feature.

        Returns:
            The point.
        """
        if node_index in self._used_point_indexes:
            raise AssertionError(f'Point at index {node_index} added twice')
        self._used_point_indexes.add(node_index)

        if feature_id != 0 and feature_id in self._used_point_feature_ids:
            raise AssertionError(f'Duplicate point feature ID: {feature_id}')
        self._used_point_feature_ids.add(feature_id)

        x, y, z = self._locations[node_index]
        point = Point(x, y, z, feature_id=feature_id)
        return point

    def add_node(self, node_index: int, component_id: int, feature_id: int = 0):
        """
        Add a point to the coverage based on a node index in the UGrid.

        Args:
            node_index: Index of the node to add. This is its index in the UGrid's point stream.
            component_id: Component ID to assign the point.
            feature_id: Feature ID to assign the point. If 0, an available one will be assigned.
        """
        if node_index in self._used_point_indexes:
            raise AssertionError(f'Point at index {node_index} added twice')
        self._used_point_indexes.add(node_index)

        if feature_id != 0 and feature_id in self._used_point_feature_ids:
            raise AssertionError(f'Duplicate point feature ID: {feature_id}')
        self._used_point_feature_ids.add(feature_id)

        x, y, z = self._locations[node_index]
        point = Point(x, y, z, feature_id=feature_id)

        self._points.append(point)
        self._point_component_ids.append(component_id)

    def add_node_string(self, node_indexes: list[int], component_id: int, feature_id: int = 0):
        """
        Add an arc to the coverage based on node indexes in the UGrid.

        Args:
            node_indexes: Indexes of nodes defining the arc.
            component_id: Component ID to assign the arc.
            feature_id: Feature ID to assign the arc. If zero (the default), the arc will be implicitly assigned an
                available ID. If nonzero, the arc will be explicitly assigned the provided ID. Explicit IDs must be
                unique. Implicit ones will never conflict with explicit ones.
        """
        if feature_id != 0 and feature_id in self._used_arc_ids:
            raise AssertionError(f'Duplicate arc ID: {feature_id}')
        self._used_arc_ids.add(feature_id)

        locations = [self._locations[index] for index in node_indexes]
        points = [Point(x, y, z) for x, y, z in locations]

        # Arc nodes need to have valid IDs. We'll assign the ID of the node in the mesh.
        # Interesting note: If we send SMS multiple points with the same location and feature ID, it merges them.
        # This means if two or more arcs meet at the same node, we don't actually have to send the same Point object.
        # Not sure if that's intentional, but it makes this simpler, so it gets exploited here.
        points[0].id = node_indexes[0] + 1
        points[-1].id = node_indexes[-1] + 1
        arc = Arc(feature_id, points[0], points[-1], points[1:-1])
        self._arcs.append(arc)
        self._arc_component_ids.append(component_id)

    def add_polygon(self, polygon: Polygon, component_id: int):
        """
        Add a polygon to the coverage.

        If the polygon's feature ID is nonzero, it must be unique among all added polygons. If it is zero, the polygon
        will be assigned an ID that does not conflict with other polygons. This assignment is deferred until after all
        the other polygons are collected, so implicitly numbered polygons will never conflict with explicit ones.

        Args:
            polygon: The polygon to add.
            component_id: Component ID to assign the polygon.
        """
        if polygon.id != 0 and polygon.id in self._used_polygon_ids:
            raise AssertionError(f'Duplicate polygon ID: {polygon.id}')

        self._used_polygon_ids.add(polygon.id)
        self._polygons.append(polygon)
        self._polygon_component_ids.append(component_id)

    def complete(self) -> Coverage:
        """
        Mark the coverage as complete.

        This is an older API. Prefer finish() instead.

        Returns:
            The completed coverage.
        """
        if self._points:
            _assign_ids(self._points)
            self._coverage.set_points(self._points)

        if self._arcs:
            _assign_ids(self._arcs)
            self._coverage.arcs = self._arcs

        if self._polygons:
            _assign_ids(self._polygons)
            self._coverage.polygons = self._polygons

        self._write_ids()

        self._coverage.complete()
        return self._coverage

    def finish(self) -> tuple[Coverage, Any]:
        """
        Mark the coverage as complete.

        The returned keywords should be passed to Query.add_coverage. They are an implementation detail. It is the
        builder's responsibility to ensure they are valid, so user code should generally just treat them as an opaque
        object and not worry about what they actually are.

        Returns:
            A tuple of (coverage, keywords). Both can be passed to Query.add_coverage.
        """
        # Components sent from ActionRequest callbacks need keywords sent, but they aren't implemented yet. This API is
        # here to give current users a consistent pattern to follow that will still work once those are implemented.
        return self.complete(), None

    def _write_ids(self):
        """Write out the ID files."""
        with DisplayOptionsHelper(self._main_file) as helper:
            if self._points:
                feature_ids = [point.id for point in self._points]
                helper.request_feature_update(TargetType.point, self._point_component_ids, feature_ids)
            if self._arcs:
                feature_ids = [arc.id for arc in self._arcs]
                helper.request_feature_update(TargetType.arc, self._arc_component_ids, feature_ids)
            if self._polygons:
                feature_ids = [polygon.id for polygon in self._polygons]
                helper.request_feature_update(TargetType.polygon, self._polygon_component_ids, feature_ids)


def _assign_ids(features: list[Point] | list[Arc] | list[Polygon]):
    """
    Assign IDs features.

    Any feature with an ID of 0 will be assigned a new feature ID. New IDs are selected starting from 1 and incrementing
    by 1 with each new ID, skipping any that are already in use. Features with nonzero IDs will be left alone, even if
    that means leaving gaps in the numbering.

    Args:
        features: List of features to assign IDs to.
    """
    used_ids = {feature.id for feature in features}
    available_ids = (i for i in count(start=1) if i not in used_ids)
    for feature in features:
        if feature.id == 0:
            feature.id = next(available_ids)
