"""Module for the MappedUpwindSolverCoverageComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.gmi.components.gmi_component import GmiComponent
from xms.guipy.dialogs.process_feedback_dlg import run_feedback_dialog

# 4. Local modules
from xms.schism.data.mapped_upwind_solver_coverage_data import MappedUpwindSolverCoverageData
from xms.schism.feedback.unmap_solver_coverage_runner import UnmapSolverCoverageRunner


class MappedUpwindSolverCoverageComponent(GmiComponent):
    """Component for a mapped upwind solver coverage."""
    def __init__(self, path: str | Path):
        """
        Initialize the component.

        Args:
            path: Path to the component's main-file.
        """
        super().__init__(path)
        self.tree_commands.append(('Convert to coverage', 'unmap_coverage'))

    def get_initial_display_options(self, query: Query, params):
        """Get initial display options."""
        main_file = Path(self.main_file)
        json_file = main_file.parent / 'poly_display.json'
        self.display_option_list = [XmsDisplayMessage(file=str(json_file), draw_type=DrawType.draw_at_locations)]
        return [], []

    def _get_data(self):
        """Get the component's data manager."""
        return MappedUpwindSolverCoverageData(self.main_file)

    def unmap_coverage(self, query: Query, _params: list[dict],
                       parent: QWidget) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        Unmap the coverage.

        Args:
            query: Interprocess communication object.
            _params: Ignored.
            parent: Parent widget.

        Returns:
            Messages and requests. Always empty.
        """
        worker = UnmapSolverCoverageRunner(self.data, query=query)
        run_feedback_dialog(worker, parent)
        return [], []
