"""Module for reading a namelist."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
from pathlib import Path

# 2. Third party modules
from f90nml import read

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Section, Type

# 4. Local modules
from xms.schism.data.model import enabled


def read_namelist(path: Path, section: Section):
    """
    Read a namelist.

    Args:
        path: Path to read the file from.
        section: Section to read the contents into.
    """
    namelist = read(path)

    for group_name in namelist:
        namelist_group = namelist[group_name]
        model_group = section.group(group_name.lower())
        for parameter_name in namelist_group:
            value = namelist_group[parameter_name]
            _apply_value(model_group, parameter_name, value)

    # If ihot=1,2 build a file path to the hotstart file. Do this after the big switch statement above so we can get
    # the hotstart option combobox populated as well
    _build_hotstart_filepath(path=path, section=section)


def _apply_value(model_group, parameter_name, value):
    """
    Convert a value from what the namelist reader provides to what GenericModel requires and assign it.

    Args:
        model_group: Group to apply the value in.
        parameter_name: Name of the parameter to apply.
        value: The value to apply.
    """
    # Namelists can specify array values by index, e.g. `flag_ic(1) = 1; flag_ic(3) = 1`.
    # If you omit an index in the middle (as in the example), the reader fills it with
    # None, so we have to check for that with arrays.
    if parameter_name == 'flag_ic':
        for index, element in enumerate(value, start=1):
            if element is not None:
                name = f'{parameter_name}({index})'
                model_group.parameter(name).value = str(element)
    # Same situation as flag_ic, just a different name.
    elif parameter_name == 'inu_tr':
        for index, element in enumerate(value, start=1):
            if value is not None:
                name = f'{parameter_name}({index})'
                model_group.parameter(name).value = str(element)
    # This is technically an array. It was made a table for reasons I'm unsure of.
    elif parameter_name == 'iof_hydro':
        values = copy.deepcopy(model_group.parameter(parameter_name).default)
        for index, row_value in enumerate(value):
            if row_value is not None:
                values[index][-1] = row_value
        model_group.parameter(parameter_name).value = values
    # Booleans come in as integers.
    elif (parameter := model_group.parameter(parameter_name)).parameter_type == Type.BOOLEAN:
        parameter.value = bool(value)
    # Most options come in as integers, except opt/mid and opt/stab, which are strings (other/executable is a string
    # too, but it isn't part of the namelist, so we don't worry about it here). We have to convert them to strings due
    # to a flaw in the option type's design. It doesn't distinguish between labels and values, so its values must be
    # strings whether we like it or not.
    elif parameter.parameter_type == Type.OPTION:
        parameter.value = str(value)
    # We'll just assume everything else is fine until it explodes.
    else:
        model_group.parameter(parameter_name.lower()).value = value


def _build_hotstart_filepath(path: Path, section: Section):
    """
    Builds the file path for a hotstart file based on the given conditions.

    Args:
        path: Path to the param.nml file.
        section: The Section object.
    """
    path = Path(path)
    if enabled(section, 'opt', 'ihot_file'):
        ihot_file = path.parent / 'hotstart.nc'
        section.group('opt').parameter('ihot_file').value = str(ihot_file)
