"""Namelist writer."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Section, Type

# 4. Local modules
from xms.schism.resources import param_nml_template


def write_namelist(section: Section, path: Path):
    """
    Write a namelist to disk.

    Args:
        section: Global values from a generic model.
        path: Where to write the file to.
    """
    with NamelistWriter(section, path) as writer:
        writer.write()


class NamelistWriter:
    """Writer class for the parameter input from SCHISM."""
    def __init__(self, section: Section, path: str | Path):
        """Constructor.

        Args:
            section: Global parameters, with values, to write.
            path: Where to write the output.
        """
        self._path = Path(path)  # Could be a string. Ensure it's a path.
        self._input_file: TextIO
        self._section = section

    def __enter__(self):
        """Open the file."""
        self._output = open(self._path, 'w')
        self._template = open(param_nml_template)
        return self

    def __exit__(self, _exc_type, _exc_value, _exc_tb):
        """Close the file."""
        self._template.close()
        self._output.close()

    def _replace(self, line: str, start: int, end: int):
        """
        Replace an occurrence of '{group/name}' with the value of the parameter.

        Args:
            line: Line to replace in.
            start: Index of the opening {.
            end: Index of the closing }.

        Returns:
            `line`, with the substitution marker replaced.
        """
        end += 1
        identifier = line[start + 1:end - 1]
        group_name, parameter_name = identifier.split('/')
        substitution = _format(self._section, group_name, parameter_name)
        substituted = line[:start] + substitution + line[end:]
        return substituted

    def write(self):
        """Write the namelist."""
        for line in self._template:
            open_position = line.find('{')
            close_position = line.find('}')
            comment_position = line.find('!')

            open_before_comment = (0 <= open_position < comment_position)
            open_without_comment = (comment_position < 0 <= open_position)

            if open_before_comment or open_without_comment:
                line = self._replace(line, open_position, close_position)
            self._output.write(line)


def _format(section: Section, group_name: str, parameter_name: str):
    """
    Write a parameter.

    Args:
        section: Section containing the parameter.
        group_name: Name of the group containing the parameter.
        parameter_name: Name of the parameter.
    """
    if parameter_name.startswith('iof_hydro'):
        parameter = section.group(group_name).parameter('iof_hydro')
        str_index = parameter_name[len('iof_hydro('):-1]  # Name is like `iof_hydro(#)`. Extract `#`.
        index = int(str_index) - 1  # Convert 1-based to 0-based
        value = parameter.value[index][-1]
        return str(value)
    else:
        parameter = section.group(group_name).parameter(parameter_name)
        parameter_type = parameter.parameter_type

        value = parameter.value
        if parameter_type == Type.FLOAT:
            value = float(value)
        elif parameter_type == Type.BOOLEAN:
            value = int(value)
        elif parameter_type == Type.OPTION:
            value = int(value)
        return str(value)
