"""Imports SRH Hydro file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg

# 4. Local modules
from xms.srh.components.import_simulation_runner import SRHSimImporter


class HydroImportRunner(QThread):
    """Worker thread for the .srhhydro import script."""
    processing_finished = Signal()

    def __init__(self):
        """Constructor."""
        super().__init__()
        self._logger = logging.getLogger('xms.srh')
        self.reader = None

    def run(self):
        """Imports an SRH-2D simulation."""
        try:
            self.reader = SRHSimImporter()
            self.reader.read()
        except:  # noqa - # pragma no cover - hard to test exceptions using QueryPlayback
            self._logger.exception('Error importing SRH-2D simulation.')
        finally:
            self.processing_finished.emit()


def import_hydro_with_feedback():
    """Import an .srhhydro file with the feedback dialog."""
    xms_dlg.ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
    win_cont = xms_dlg.get_parent_window_container(parent_hwnd)
    # Create the timer that keeps our Python dialog in the foreground of XMS.
    _ = win_gui.create_and_connect_raise_timer(main_hwnd, win_cont)  # Keep the timer in scope  # noqa

    worker = HydroImportRunner()
    display_text = {
        'title': 'SRH-2D Simulation Import',
        'working_prompt': 'Importing SRH-2D simulation. Please wait...',
        'warning_prompt': 'Warning(s) encountered reading the SRH-2D simulation. Review log output for more details.',
        'error_prompt': 'Error(s) encountered reading the SRH-2D simulation. Review log output for more details.',
        'success_prompt': 'Successfully imported SRH-2D simulation',
        'note': '',
        'auto_load': 'Auto load simulation into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.srh', worker, win_cont)
    feedback_dlg.exec()
    win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
    if not LogEchoQSignalStream.logged_error:  # Only send if no error.
        worker.reader.send()  # Send data back to XMS
