"""Parameter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt

# 3. Aquaveo modules
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate

# 4. Local modules
from xms.srh.components.parameters.parameter import Parameter


class BcParameter(Parameter):
    """Represents a parameter used in ParametersManager and the Advanced Simulation dialog."""
    def __init__(
        self,
        use=0,
        id_number=-1,
        type='',
        description='',
        default=0.0,
        string_value='',
        value=0.0,
        optimized_value=0.0,
        min=0.0,
        max=0.0
    ):
        """Initializes the class.

        Args:
            use (int): 1 if using this parameter, 0 if not
            id_number (int): Number used in the id_string.
            type (str): E.g. "Manning's N", 'Inlet Q', 'Exit H', 'Time step', 'Initial condition'
            description (str): Typically the arc number, or material name.
            default (float): default value
            string_value (str): string value
            value (float): value
            optimized_value (float): optimized value
            min (float): min
            max (float): max
        """
        super().__init__(use, id_number, type, description, default, string_value, value, optimized_value, min, max)

    def id_string(self):
        """Returns a string for use in the 'id' column.

        Returns:
            (str): See description.
        """
        if self.type == 'Exit H':
            return f'exit_h_{self.id_number}'
        elif self.type == 'Internal sink':
            return f'internal_sink_{self.id_number}'

    def add_run_data(self, disk_data, run_count, param_data):
        """Adds columns to the runs table.

        If disk_data has information about this parameter, uses it.

        Args:
            param_data (ParameterData): Param data.
            run_count (int): Number of runs.
            disk_data (dict): Param data from file.
        """
        base_name = self.column_base_name()
        self._add_column(disk_data, run_count, f'{base_name} Transient', object, param_data)
        self._add_column(disk_data, run_count, base_name, 'float', param_data)
        self._add_column(disk_data, run_count, f'{base_name} XY Series', object, param_data)

    def column_delegate(self, column_name, xy_series_names, dialog):
        """Give the name of a column in the runs table, returns the delegate, or None.

        Args:
            column_name (str): The name of the column in the runs table.
            xy_series_names (list): List of the names of the xy series.
            dialog: Parent of the delegate.

        Returns:
            The delegate or None
        """
        delegate = None
        if column_name.endswith('Transient'):
            delegate = QxCbxDelegate(dialog)
            delegate.set_strings(['Constant', 'Transient'])
        elif column_name.endswith('XY Series'):
            delegate = QxCbxDelegate(dialog)
            delegate.set_strings(xy_series_names)
        return delegate

    def column_default(self, column_name):
        """Given a column name, return the default value.

        Args:
            column_name (str): The name of the column in the runs table.

        Returns:
            Default value.
        """
        if column_name.endswith('XY Series'):
            return ''
        elif column_name.endswith('Transient'):
            return 'Constant'
        else:  # Constant value column
            return self.default

    def flags(self, index, column_name, flags):
        """Returns the flags for an item in the model.

        Args:
            index (QModelIndex): The item's model index.
            column_name (str): Name of the column.
            flags (Qt.ItemFlags): The flags in their initial state

        Returns:
            (Qt.ItemFlags): Updated flags for the item
        """
        if column_name.endswith('XY Series'):
            model = index.model()
            col_offset = 2  # The Transient combo box column is 2 columns to the left
            transient_cbx_value = model.data(model.index(index.row(), index.column() - col_offset))
            if transient_cbx_value == 'Constant':
                flags = flags & (~Qt.ItemIsEditable)  # remove editable flag
                flags = flags & (~Qt.ItemIsEnabled)  # remove enabled flag
        elif column_name.endswith('Transient'):
            pass
        else:  # Constant value column
            model = index.model()
            col_offset = 1  # The Transient combo box column is 1 column to the left
            transient_cbx_value = model.data(model.index(index.row(), index.column() - col_offset))
            if transient_cbx_value == 'Transient':
                flags = flags & (~Qt.ItemIsEditable)  # remove editable flag
                flags = flags & (~Qt.ItemIsEnabled)  # remove enabled flag

        return flags
