"""StatisticianRunner class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg
from xms.guipy.dialogs.process_feedback_thread import ProcessFeedbackThread

# 4. Local modules
from xms.srh.model.statistician import Statistician


class StatisticianRunner(ProcessFeedbackThread):
    """Worker thread for running solution dataset statistical analysis."""
    def __init__(self, parent, query):
        """Construct the worker.

        Args:
            parent(QWidget): parent widget
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS

        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._query = query
        self._logger = logging.getLogger('xms.srh')
        self.statistician = None

    def _do_work(self):
        """Run statistical analysis."""
        try:
            self.statistician = Statistician(query=self._query)
            self.statistician.run_statistics()
        except:  # noqa - # pragma no cover - hard to test exceptions using QueryPlayback
            self._logger.exception('Error running statistical analysis on SRH-2D simulation solution.')
        finally:
            self.processing_finished.emit()


def run_statistics_with_feedback(query, win_cont):
    """Run statistical analysis on a simulation solution.

    Args:
        query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
        win_cont (:obj:`object`): Qt parent
    """
    worker = StatisticianRunner(win_cont, query)
    display_text = {
        'title': 'SRH-2D Solution Statistics',
        'working_prompt': 'Running statistical analysis on SRH-2D solution. Please wait...',
        'error_prompt': 'Error(s) encountered running statistical analysis. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered running statistical analysis. Review log output for more details.',
        'success_prompt': 'Successfully computed solution statistics.',
        'note': '',
        'auto_load': 'Auto load statistical data sets into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.srh', worker, win_cont)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    feedback_dlg.exec()
    # win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
    if not LogEchoQSignalStream.logged_error:  # Only send if no error.
        worker.statistician.send()  # Send data back to XMS
