"""ModelControlHydro class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler

# 4. Local modules


class ModelControlHydro(param.Parameterized):
    """Manages data file for the hidden simulation component."""
    # Hydrodynamics
    simulation_description = param.String(
        default='Description',
        doc='description field for input',
        precedence=1,
    )
    case_name = param.String(
        default='Case',
        doc='case name input field',
        precedence=2,
    )
    start_time = param.Number(
        label='Start time (hours)',
        default=0.0,
        bounds=(0.0, None),
        doc='simulation start time',
        precedence=3,
    )
    time_step = param.Number(
        label='Time step (seconds)',
        default=1.0,
        bounds=(0.0, None),
        doc='simulation time step',
        precedence=4,
    )
    end_time = param.Number(
        label='End time (hours)',
        default=1.0,
        bounds=(0.0, None),
        doc='simulation end time',
        precedence=5,
    )
    initial_condition = param.ObjectSelector(
        label='',
        default='Dry',
        objects=[
            'Dry', 'Initial Water Surface Elevation', 'Automatic', 'Restart File', 'Water Surface Elevation Dataset'
        ],
        doc='simulation initial condition',
        precedence=6,
    )
    restart_file = param.String(
        # path='*.rst',
        doc='restart file for simulation initial condition',
        precedence=6.1,
    )
    initial_water_surface_elevation = param.Number(
        default=0.0,
        doc='water surface elevation for simulation initial condition',
        precedence=6.2,
    )
    initial_water_surface_elevation_units = param.ObjectSelector(
        label='',
        default='Feet',
        objects=['Feet', 'Meters'],
        doc='units for initial water surface elevation',
        precedence=6.3,
    )
    select_wse_dataset = param.Action(
        doc='Selector for the wse dataset.',
        label='Select Dataset...',
        precedence=6.4,
    )
    water_surface_elevation_dataset = param.String(
        label='',
        default='No Dataset selected',
        doc='UUID of dataset selected from xms',  # Time step number (1-based) may be appended with a colon ('uuid:ts')
        precedence=6.5,
    )
    use_pressure_ceiling = param.Boolean(
        default=False,
        doc='',
        precedence=7,
    )
    generate_pressure_ceiling = param.Action(
        doc='Generates SRH ceiling file from 3d bridge definitions.',
        label='Generate new pressure ceiling file...',
        precedence=7.1,
    )
    ceiling_file_label = param.String(
        doc='String to indicate to the user if they have a ceiling file.',
        default='No ceiling file defined',
        label='',
        precedence=7.2,
    )

    def __init__(self):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.

        """
        super().__init__()
        self.on_update_initial_condition = None
        self.water_surface_elevation_dataset = 'No Dataset selected'
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['initial_condition', 'use_pressure_ceiling'],
            depends={
                'Initial Water Surface Elevation':
                    ['initial_water_surface_elevation', 'initial_water_surface_elevation_units'],
                'Restart File': ['restart_file'],
                'Water Surface Elevation Dataset': ['select_wse_dataset', 'water_surface_elevation_dataset'],
                'use_pressure_ceiling': ['generate_pressure_ceiling', 'ceiling_file_label'],
            },
        )

    @param.depends('initial_condition', watch=True)
    def _update_initial_condition(self):
        """Watches changes made to the initial_condition object."""
        self.enabler.do_enabling(force_disable=False)
        if self.on_update_initial_condition is not None:
            self.on_update_initial_condition()

    @param.depends('use_pressure_ceiling', watch=True)
    def _update_use_pressure_ceiling(self):
        """Watches changes made to the use_pressure_ceiling object."""
        self.enabler.do_enabling(force_disable=False)

    # @param.depends('initial_water_surface_elevation_units', watch=True)
    # def _update_initial_water_surface_elevation_units(self):
    #     """watches changes made to the initial_water_surface_elevation_units object
    #     """
    #     from decimal import Decimal
    #     from decimal import getcontext
    #     # transform the units from meters to feet or back
    #     if self.initial_water_surface_elevation_units == 'Meters':
    #         prec = getcontext().prec
    #         # this works for converting between 1 meter to feet and then back again
    #         getcontext().prec = 15
    #         val = self.initial_water_surface_elevation
    #         val = Decimal(val) * Decimal(0.3048)
    #         self.initial_water_surface_elevation = float(val)
    #         getcontext().prec = prec
    #     else:
    #         val = self.initial_water_surface_elevation
    #         val = Decimal(val) / Decimal(0.3048)
    #         self.initial_water_surface_elevation = float(val)
    #     self.enabler.do_enabling(force_disable=False)
