"""BcDataInletSc class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler  # noqa: I202

# 4. Local modules
from xms.srh.data.par.bc_data_inlet_sc_wse import BcDataInletScWse


class BcDataInletSc(param.Parameterized):
    """Bc type inlet-sc data."""
    discharge_q_option = param.ObjectSelector(
        label='Discharge Q option',
        default='Constant',
        objects=['Constant', 'Time series'],
        doc='option for specifying the inflow',
        precedence=1,
    )
    discharge_q_units = param.ObjectSelector(
        label='Discharge Q units',
        default='English',
        objects=['English', 'Metric'],
        doc='units for discharge',
        precedence=2,
    )
    constant_q = param.Number(
        label='Discharge (Q)',
        default=0.0,
        doc='constant flow',
        precedence=3,
    )
    constant_q_units = param.String(
        label='',
        default='cfs',
        doc='constant flow units',
        precedence=4,
    )
    constant_wse = param.Number(
        label='Water elevation (WSE)',
        default=0.0,
        doc='constant wse',
        precedence=5,
    )
    constant_wse_units = param.String(
        label='',
        default='feet',
        doc='constant wse units',
        precedence=6,
    )

    time_series_q = param.DataFrame(
        label='Time series Q',
        default=pd.DataFrame(columns=['hrs', 'vol_per_sec']),
        doc='time series for Q',
        precedence=7,
    )
    time_series_q_units = param.String(
        label='',
        default='hrs -vs- cfs',
        doc='units for time series Q',
        precedence=8,
    )

    water_surface_elevation = param.ClassSelector(
        label='',
        class_=BcDataInletScWse,
        precedence=9,
    )

    distribution_at_inlet = param.ObjectSelector(
        label='Distribution at inlet',
        default='Conveyance',
        objects=['Conveyance', 'Profile', 'Q', 'Velocity'],
        doc='',
        precedence=10,
    )

    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self.water_surface_elevation = BcDataInletScWse()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['discharge_q_option'],
            depends={
                'Constant': ['constant_q', 'constant_q_units', 'constant_wse', 'constant_wse_units'],
                'Time series': ['time_series_q', 'time_series_q_units', 'water_surface_elevation'],
            },
        )

    @param.depends('discharge_q_option', watch=True)
    def _update_discharge_q_option(self):
        """Watches changes made to the discharge_option object."""
        self.enabler.do_enabling(force_disable=False)

    @param.depends('discharge_q_units', watch=True)
    def _update_discharge_q_units(self):
        """Watches changes made to the discharge_q_units object."""
        if self.discharge_q_units == 'English':
            self.constant_q_units = 'cfs'
            self.constant_wse_units = 'feet'
            self.time_series_q_units = 'hrs -vs- cfs'
            self.water_surface_elevation.time_series_wse_units = 'hrs -vs- feet'
            self.water_surface_elevation.rating_curve_units = 'cfs -vs- feet'
        else:
            self.constant_q_units = 'cms'
            self.constant_wse_units = 'meters'
            self.time_series_q_units = 'hrs -vs- cms'
            self.water_surface_elevation.time_series_wse_units = 'hrs -vs- meters'
            self.water_surface_elevation.rating_curve_units = 'cfs -vs- meters'
        self.enabler.do_enabling(force_disable=False)
