"""BcDataPressure class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler  # noqa: I202

# 4. Local modules
from xms.srh.data.par.bc_data_weir_type import BcDataWeirType


class BcDataPressure(param.Parameterized):
    """Bc type pressure data."""
    ceiling_type = param.ObjectSelector(
        default='Flat',
        objects=['Flat', 'Parabolic'],
        doc='',
        precedence=1,
    )
    units = param.ObjectSelector(
        default='Feet',
        objects=['Feet', 'Meters'],
        doc='',
        precedence=2,
    )
    upstream_elevation = param.Number(
        label='Ceiling elevation along upstream (Flat) or edges (Parabolic)',
        default=0.0,
        doc='',
        precedence=3,
    )
    downstream_elevation = param.Number(
        label='Ceiling elevation along downstream (Flat) or midspan (Parabolic)',
        default=0.0,
        doc='',
        precedence=4,
    )
    roughness = param.Number(
        label='Manning roughness coefficient between water and ceiling',
        default=0.0,
        doc='',
        precedence=5,
    )
    overtopping = param.Boolean(
        label='Model overtopping on this zone',
        default=False,
        doc='',
        precedence=6,
    )
    weir_crest_elevation = param.Number(
        label='Crest elevation',
        default=0.0,
        doc='',
        precedence=7,
    )
    weir_length = param.Number(
        label='Length of weir over pressure zone',
        default=0.0,
        doc='',
        precedence=8,
    )
    weir_type = param.ClassSelector(
        label='',
        class_=BcDataWeirType,
        doc='',
        precedence=9,
    )
    total_head = param.Boolean(
        label='Use total head (velocity and water surface)',
        default=False,
        doc='',
        precedence=10,
    )

    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self.weir_type = BcDataWeirType()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['overtopping'],
            depends={
                'overtopping': ['weir_crest_elevation', 'weir_length', 'weir_type', 'total_head'],
            },
        )

    @param.depends('overtopping', watch=True)
    def _update_overtopping(self):
        """Watches changes made to the overtopping object."""
        self.enabler.do_enabling(force_disable=False)
