"""SedimentEquationParameters class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler

# 4. Local modules


class SedimentEquationParameters(param.Parameterized):
    """Manages data file for the hidden simulation component."""
    # Sediment transport
    sediment_transport_equation = param.ObjectSelector(
        default='Engelund-Hansen',
        objects=[
            'Engelund-Hansen', 'Meyer-Peter-Muller', 'Parker', 'Wilcock-Crowe', 'Wu-et-al',
            'Yang 1973 Sand w 1984 Gravel', 'Yang 1979 Sand w 1984 Gravel', 'Mixed'
        ],
        doc='transport equation',
        precedence=1,
    )
    meyer_peter_muller_hiding_factor = param.Number(
        default=0.0,
        bounds=(0.0, None),
        doc='hiding factor',
        precedence=1.1,
    )
    parker_reference_shields_parameter = param.Number(
        default=0.04,
        bounds=(0.0, None),
        doc='',
        precedence=1.2,
    )
    parker_hiding_coefficient = param.Number(
        default=0.65,
        bounds=(0.0, None),
        doc='',
        precedence=1.3,
    )
    wilcox_t1_coefficient = param.Number(
        label='Wilcock T1 coefficient',
        default=0.021,
        bounds=(0.0, None),
        doc='wilcock T1 value',
        precedence=1.4,
    )
    wilcox_t2_coefficient = param.Number(
        label='Wilcock T2 coefficient',
        default=0.036,
        bounds=(0.0, None),
        doc='wilcock T2 value',
        precedence=1.5,
    )
    wilcox_sand_diameter = param.Number(
        label='Wilcock sand diameter',
        default=1.0,
        bounds=(0.0, None),
        doc='wilcock sand diameter',
        precedence=1.6,
    )
    wu_critical_shields_parameter = param.Number(
        default=0.03,
        bounds=(0.0, None),
        doc='wu shields parameter',
        precedence=1.7,
    )
    mixed_sediment_size_class_cutoff = param.Number(
        label='Sediment size class cutoff (note: the size class is the table number minus 1 for the threshold '
        'particle size)',
        default=0.0,
        bounds=(0.0, None),
        doc='mixed sediment size cutoff',
        precedence=1.8,
    )
    lower_transport_parameters = param.ClassSelector(
        label='',
        # is an instance of SedimentEquationParameters but can't declare that here
        class_=param.Parameterized,
        doc='The lower size class sediment transport equation',
        precedence=2,
    )
    higher_transport_parameters = param.ClassSelector(
        label='',
        # is an instance of SedimentEquationParameters but can't declare that here
        class_=param.Parameterized,
        doc='The higher size class sediment transport equation',
        precedence=3,
    )

    def __init__(self, include_mixed):
        """Initializes the data class.

        Args:
            include_mixed (:obj:`bool`): flag for type of class to create.

        """
        super().__init__()
        self.include_mixed = include_mixed
        if self.include_mixed:
            lst = self.param.sediment_transport_equation.objects.copy()
            self.param.sediment_transport_equation.objects = lst
            self.lower_transport_parameters = SedimentEquationParameters(include_mixed=False)
            self.higher_transport_parameters = SedimentEquationParameters(include_mixed=False)
        else:
            lst = self.param.sediment_transport_equation.objects[:-1]
            self.param.sediment_transport_equation.objects = lst

        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['sediment_transport_equation'],
            depends={
                'Meyer-Peter-Muller': ['meyer_peter_muller_hiding_factor'],
                'Parker': ['parker_reference_shields_parameter', 'parker_hiding_coefficient'],
                'Wilcock-Crowe': ['wilcox_t1_coefficient', 'wilcox_t2_coefficient', 'wilcox_sand_diameter'],
                'Wu-et-al': ['wu_critical_shields_parameter'],
                'Mixed':
                    ['mixed_sediment_size_class_cutoff', 'lower_transport_parameters', 'higher_transport_parameters'],
            },
        )

    @param.depends('sediment_transport_equation', watch=True)
    def _update_sediment_transport_equation(self):
        """Watches changes made to the sediment_transport_equation object."""
        self.enabler.do_enabling(force_disable=False)
