"""SedimentParameters class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler

# 4. Local modules


class SedimentParameters(param.Parameterized):
    """Manages data file for the hidden simulation component."""
    water_temperature = param.Number(
        default=25.0,
        bounds=(0.0, None),
        doc='temperature in degrees C',
        precedence=1,
    )
    deposition_coefficient = param.Number(
        default=0.25,
        bounds=(0.0, None),
        doc='',
        precedence=2,
    )
    erosion_coefficient = param.Number(
        default=1.0,
        bounds=(0.0, None),
        doc='',
        precedence=3,
    )

    adaptation_length_bedload_mode = param.ObjectSelector(
        label='Mode',
        default='Constant Length',
        objects=[
            'Constant Length', 'Philips-Sutherland Saltation Length Formula', 'Van Rijn Dune Formula',
            'Van Rijn Saltation Length Formula', 'Seminara et al. (2002) Formula'
        ],
        doc='',
        precedence=4,
    )
    adaptation_length_bedload_length = param.Number(
        label='Length',
        default=0.000001,
        bounds=(0.0, None),
        doc='',
        precedence=4.1,
    )

    active_layer_thickness_mode = param.ObjectSelector(
        label='Mode',
        default='Constant Thickness',
        objects=['Constant Thickness', 'Thickness based on D90'],
        doc='',
        precedence=5,
    )
    active_layer_constant_thickness = param.Number(
        label='Thickness (mm)',
        default=1.0,
        bounds=(0.0, None),
        doc='',
        precedence=5.1,
    )
    active_layer_d90_scale = param.Number(
        label='D90 thickness scale',
        default=1.0,
        bounds=(0.0, None),
        doc='',
        precedence=5.2,
    )

    def __init__(self):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.

        """
        super().__init__()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['adaptation_length_bedload_mode', 'active_layer_thickness_mode'],
            depends={
                'Constant Length': ['adaptation_length_bedload_length'],
                'Constant Thickness': ['active_layer_constant_thickness'],
                'Thickness based on D90': ['active_layer_d90_scale']
            }
        )

    @param.depends('adaptation_length_bedload_mode', watch=True)
    def _update_adaptation_length_bedload_mode(self):
        self.enabler.do_enabling(force_disable=False)

    @param.depends('active_layer_thickness_mode', watch=True)
    def _update_active_layer_thickness_mode(self):
        self.enabler.do_enabling(force_disable=False)
