"""CoverageLooperBase class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import CoverageItem
from xms.api.tree import tree_util

# 4. Local modules


class CoverageLooperBase:
    """Common code for adding coverages to the report.

    Follows the Template Method Pattern. visit_coverages is the main function that does not need to be overridden.
    Override the other methods as necessary.
    """
    def __init__(
        self,
        notes_db,
        report_dir,
        query,
        coverage_type,
        component_unique_name,
        model_name='SRH-2D',
        mesh_boundaries=None,
        coverage_meshes=None
    ):  # pragma: no cover
        """Initializes the class.

        Args:
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Path to directory where report files are created.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            coverage_type (:obj:`str`): Take name from xml
            component_unique_name (:obj:`str`): Unique name from xml
            model_name (:obj:`str`): Name of the coverage's XML definition model
            mesh_boundaries (:obj:`dict{mesh UUID: MeshBoundaryTuple}`): The mesh boundaries.
            coverage_meshes (:obj:`dict{coverage UUID: mesh UUID}`): Coverages and their meshes.
        """
        self._notes_db = notes_db
        self._report_dir = report_dir
        self._query = query
        self._model_name = model_name
        self._coverage_type = coverage_type
        self._coverage_unique_name = component_unique_name
        self._mesh_boundaries = mesh_boundaries
        self._coverage_meshes = coverage_meshes

        self._coverages_jinja_list = []  # List of dicts, 1 per coverage, containing jinja data

    def visit_coverages(self):  # pragma: no cover
        """Visit all coverages, processing those that match the type we're looking for.

        Don't override this one.

        Returns:
            (:obj:`list[dict]`):List of dicts size of number of coverages processed containing jinja data.
        """
        # Go through all coverages processing the obstruction coverages
        map_data = tree_util.child_from_name(self._query.project_tree, 'Map Data')
        coverages = tree_util.descendants_of_type(map_data, CoverageItem)
        for coverage_node in coverages:
            if coverage_node.model_name != self._model_name or coverage_node.coverage_type != self._coverage_type:
                continue

            # Get coverage and component
            coverage = self._query.item_with_uuid(coverage_node.uuid)
            data_objects_component = self._query.item_with_uuid(
                coverage_node.uuid, model_name=self._model_name, unique_name=self._coverage_unique_name
            )
            component = self._create_component(data_objects_component.main_file)
            self._load_component_ids(component)
            coverage_jinja = self._store_coverage_data(coverage, component)
            self._coverages_jinja_list.append(coverage_jinja)
        return self._coverages_jinja_list

    def _create_component(self, main_file):  # pragma: no cover
        """Constructs and returns the SRH component given its main file.

        CAN be overridden.

        Args:
            main_file: The main file associated with this component.

        Returns:
            See description.

        """
        return None

    def _load_component_ids(self, srh_component):  # pragma: no cover
        """Loads the feature arc/point IDs into the component.

        CAN be overridden.

        Args:
            srh_component: The SRH component to load the IDs into.
        """
        pass

    def _store_coverage_data(self, coverage, srh_component):  # pragma: no cover
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        CAN be overridden.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            srh_component: The SRH component.

        Returns:
            (:obj:`dict`): A dict of the coverage data for use with jinja.
        """
        return None

    @staticmethod
    def mesh_boundary_from_coverage(coverage_uuid, coverage_meshes, mesh_boundaries):  # pragma: no cover
        """Returns the mesh boundary, if one exists, associated with the coverage.

        Args:
            coverage_uuid (:obj:`str`): UUID of the coverage.
            coverage_meshes ((:obj:`dict{coverage UUID: mesh UUID}`): Coverages and their meshes.
            mesh_boundaries ((:obj:`dict{mesh UUID: MeshBoundaryTuple}`): The mesh boundaries.

        Returns:
            mesh_boundary (:obj:`MeshBoundaryTuple`): The mesh boundary.
        """
        if coverage_meshes and coverage_uuid in coverage_meshes:
            mesh_uuid = coverage_meshes[coverage_uuid]
            if mesh_uuid in mesh_boundaries:
                return mesh_boundaries[mesh_uuid]
        return None


def label_from_bc_type(bc_type):  # pragma: no cover
    """Returns a shorter label for use on the plot given the bc_type.

    Args:
        bc_type (:obj:`str`): BC type string.

    Returns:
        See description.
    """
    pos = bc_type.find(' (')
    if pos >= 0:
        return bc_type[0:pos]
    return bc_type
