"""Dialog for assigning obstruction attributes to points and arcs."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (QDialogButtonBox, QHBoxLayout, QLabel, QMessageBox, QScrollArea, QVBoxLayout, QWidget)

# 3. Aquaveo modules
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.param.param_qt_helper import ParamQtHelper

# 4. Local modules
from xms.srh.data.par.obstruction_param import ObstructionParam


class ObstructionDialog(XmsDlg):
    """Dialog for assigning obstruction attributes to points and arcs."""
    def __init__(self, win_cont, data):
        """Initializes the class, sets up the ui, and writes the model control values.

        Args:
            win_cont (:obj:`QWidget`): Parent window
            data (:obj:`ModelControl`): SRH model control class
        """
        super().__init__(win_cont, 'xms.srh.gui.obstruction_dialog')
        self.data = data
        self.set_data_layout()
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D_Coverages#Obstructions_Coverage'
        self.param_helper = ParamQtHelper(self)
        self.widgets = dict()

        self.setWindowTitle('SRH2D Assign Obstruction')
        self.setup_ui()
        self.adjustSize()
        h = max(self.size().height(), 300)
        self.resize(self.size().width() * 1.5, h)

    def set_data_layout(self):
        """Sets layout parameters for use with ParamQtHelper."""
        pass
        # self.data.param_layout = obstruction_layout()

    def setup_ui(self):
        """Initialize dialog widgets."""
        self._set_layout('', 'top_layout', QVBoxLayout())
        self.widgets['scroll_area'] = QScrollArea(self)
        self.widgets['top_layout'].addWidget(self.widgets['scroll_area'])
        self.widgets['scroll_area'].setWidgetResizable(True)
        self.widgets['scroll_area'].setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        # needed for the scroll area to function properly (see stackoverflow)
        self.widgets['scroll_area_widget'] = QWidget()
        self._set_layout('scroll_area_widget', 'scroll_area_layout', QVBoxLayout())
        self.widgets['scroll_area'].setWidget(self.widgets['scroll_area_widget'])
        self.widgets['info_message'] = QLabel('Bottom elevation specified as Z-value of arc/point/node.')
        self.widgets['scroll_area_layout'].addWidget(self.widgets['info_message'])
        self.param_helper.add_params_to_layout(self.widgets['scroll_area_layout'], self.data)
        self.widgets['scroll_area_layout'].addStretch()

        self.widgets['btn_box'] = QDialogButtonBox()
        self.widgets['top_layout'].addWidget(self.widgets['btn_box'])

        # set all widget values and hide/show
        self.param_helper.do_param_widgets(None)

        # QDialogButtonBox with Ok and Cancel buttons
        self.widgets['btn_box'].setOrientation(Qt.Horizontal)
        self.widgets['btn_box'].setStandardButtons(
            QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help
        )
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(super().reject)
        self.widgets['btn_box'].helpRequested.connect(self.help_requested)

    def _set_layout(self, parent_name, layout_name, layout):
        """Adds a layout to the parent.

        Args:
            parent_name (:obj:`str`): Name of parent widget in self.widgets or '' for self
            layout_name (:obj:`QLay`): Name of layout in parent widget
            layout (:obj:`str`): QtLayout to be used
        """
        self.widgets[layout_name] = layout
        if parent_name:
            parent = self.widgets[parent_name]
        else:
            parent = self
        if type(parent) in [QVBoxLayout, QHBoxLayout]:
            parent.addLayout(self.widgets[layout_name])
        else:
            parent.setLayout(self.widgets[layout_name])

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)


def run_dlg(sel_ids, target_type, obstruction_comp, win_cont):
    """Runs the assign Obstruction dialog.

    Args:
        sel_ids (:obj:`list`): ids of selected arcs
        target_type (:obj:`xms.guipy.data.target_type.TargetType`): One of: TargetType.point, TargetType.arc
        obstruction_comp (:obj:`ObstructionComponent`): component for bc coverage
        win_cont (:obj:`QWidget`): Parent window
    """
    type_str = 'arcs'
    if target_type == TargetType.point:
        type_str = 'points/nodes'

    comp_ids = set()
    for id in sel_ids:
        comp_id = obstruction_comp.get_comp_id(target_type, id)
        if comp_id is None:
            comp_ids.add(-1)
        elif comp_id >= 0:
            comp_ids.add(comp_id)
    # if there is more than 1 component id then we need to notify user
    if len(comp_ids) > 1:
        msg = f'The selected {type_str} are assigned different properties. Would you like to continue and assign the' \
              f' same properties to all selected {type_str}?'
        rval = QMessageBox.question(win_cont, 'Selected Features', msg, QMessageBox.Yes | QMessageBox.No)
        if rval == QMessageBox.No:
            return

    id = -1
    for first_id in comp_ids:
        id = first_id
        break
    data = ObstructionParam()
    if id != -1:
        data = obstruction_comp.data.obstruction_data_param_from_id(id)

    dlg = ObstructionDialog(win_cont, data)
    if dlg.exec():
        id = max(obstruction_comp.data.obstruction_data.to_dataframe()['id']) + 1
        obstruction_comp.data.append_obstruction_data_with_id(data, id)
        obstruction_comp.data.commit()
        for f_id in sel_ids:
            obstruction_comp.update_component_id(target_type, f_id, int(id))
