"""Class for managing interprocess communication with XMS."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os.path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util, TreeNode
from xms.data_objects.parameters import Component
from xms.gmi.data.sim_data import SimData

# 4. Local modules
from xms.srh_swmm_coupler.file_io import util


class CouplerData:
    """Class for managing interprocess communication with XMS."""

    def __init__(self, query=None):
        """Constructor.

        Args:
            query (:obj:`Query`): The XMS interprocess communication object
        """
        self._query = query or Query()
        self._logger = util.get_logger()
        self._swmm_sim = None
        self._sim_uuid = None
        self._sim_data = None
        self._srh_sim = None

    @property
    def query(self):
        """Returns the xmsapi interprocess communication object.

        Notes:
            If constructed in a component runner script, Query should be passed in on construction. If constructed in
            an export or import script, creation of the Query can happen here. Only one Query/CouplerData object should
            be constructed per script.
        """
        if self._query is None:
            self._query = Query()
        return self._query

    @property
    def sim_uuid(self):
        """Returns the simulation's UUID (not it's component's UUID)."""
        if self._sim_uuid is None:
            tree = self._query.project_tree

            # We should be at either the simulation or its hidden component. The tree should contain the simulation, but
            # not its hidden component, so we can figure out where we are by checking if the current item's UUID is in
            # the tree. If it is, we must be at the simulation. If not, we're at its component, and its parent item is
            # the simulation.
            current_item_uuid = self.query.current_item_uuid()
            node = tree_util.find_tree_node_by_uuid(tree, current_item_uuid)
            if node is not None:
                return node.uuid

            parent_item_uuid = self.query.parent_item_uuid()
            node = tree_util.find_tree_node_by_uuid(tree, parent_item_uuid)
            self._sim_uuid = node.uuid
        return self._sim_uuid

    @property
    def sim_data(self):
        """Returns the simulation component's SimData."""
        if self._sim_data is None:
            self._logger.info('Retrieving simulation data from SMS...')
            do_comp = self.query.item_with_uuid(self.sim_uuid,
                                                model_name='SRH_SWMM_COUPLER',
                                                unique_name='SimComponent')
            if not do_comp:
                raise RuntimeError(
                    'Unable to retrieve simulation data from SMS.')
            self._sim_data = SimData(do_comp.main_file)
        return self._sim_data

    @property
    def component_folder(self):
        """Returns the simulation's UUID (not it's component's UUID)."""
        return os.path.dirname(os.path.dirname(self.sim_data.main_file))

    @property
    def swmm_sim(self) -> tuple[TreeNode, Component]:
        """Returns the SWMM simulation linked to the SRH_SWMM_COUPLER sim."""
        if self._swmm_sim is not None:
            return self._swmm_sim
        self._swmm_sim = self._get_sim('SWMM', 'SimComponent')
        return self._swmm_sim

    @property
    def srh_sim(self) -> tuple[TreeNode, Component]:
        """Returns the SRH2D simulation linked to the SRH_SWMM_COUPLER sim."""
        if self._srh_sim is not None:
            return self._srh_sim
        self._srh_sim = self._get_sim('SRH-2D', 'Sim_Manager')
        return self._srh_sim

    def _get_sim(self, model_name: str, unique_name: str) -> tuple[TreeNode, Component]:
        sim_item = tree_util.find_tree_node_by_uuid(self._query.project_tree,
                                                    self.sim_uuid)
        sim_items = tree_util.descendants_of_type(tree_root=sim_item,
                                                  xms_types=['TI_DYN_SIM_PTR'],
                                                  allow_pointers=True,
                                                  model_name=model_name)
        pt = self._query.project_tree
        model_sim = None
        for sim_ptr in sim_items:
            sim = tree_util.find_tree_node_by_uuid(pt, sim_ptr.uuid)
            sim_comp = self._query.item_with_uuid(sim.uuid,
                                                  model_name=model_name,
                                                  unique_name=unique_name)
            model_sim = sim, sim_comp
        return model_sim
