"""Script used to run SRH."""
# 1. Standard python modules
import glob
import os
from pathlib import Path
import shutil

# 2. Third party modules
import h5py

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, ExecutableCommand, XmsEnvironment as XmEnv
from xms.components.bases.run_base import RunBase
from xms.core.filesystem import filesystem
from xms.data_objects.parameters import Dataset
from xms.guipy.settings import SettingsManager
from xms.srhw_exe import get_executable_paths

# 4. Local modules
from xms.srhw.components.sim_query_helper import SimQueryHelper

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"


def _get_sms_srhw_exes():
    """Get the SMS SRH exes from the registry.

    Returns:
        (:obj:`dict`): SRH exes
    """
    exe_list = get_executable_paths()
    pre_exe = ''
    srh_exe = ''
    for exe in exe_list:
        exe_lower = exe.lower()
        if exe_lower.endswith('srh-pre_v400_beta.exe'):
            pre_exe = exe
        elif exe_lower.endswith('srh-w_v100.exe'):
            srh_exe = exe
    exes = {}
    settings = SettingsManager(python_path=False)
    package = 'File_Location_Preferences'
    key = 'SRH-W - PreSRH-W'
    val = settings.get_setting(package, key, pre_exe)
    val = val if val != 'NONE' else pre_exe
    exes[key] = val
    key = 'SRH-W - SRH-W'
    val = settings.get_setting(package, key, srh_exe)
    val = val if val != 'NONE' else srh_exe
    exes[key] = val
    return exes


class RunSRHW(RunBase):
    """Says what SRH-W executables will run."""

    def __init__(self, dummy_mainfile=''):
        """Constructor."""
        super().__init__()
        self.proj_name = ''
        self.case_name = ''
        self.query_helper = None
        self._model_order = 0
        self._hydro_file = ''
        self._file_location = ''
        self._exes = []

    def _setup_query_helper(self, query):
        """Sets up the SimQueryHelper.

        Args:
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with SMS.
        """
        if not self.query_helper:
            self.query_helper = SimQueryHelper(query, at_sim=True)
            self.query_helper.get_sim_data()

    def get_executables(self, sim, query, file_location):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with SMS.
            file_location (:obj:`str`): The location of input files for the simulation.

        Returns:
            (:obj:`list[xms.api.dmi.ExecutableCommand]`):
                The executable objects to run and the action requests that go with it.
        """
        self._setup_query_helper(query)
        if not self.proj_name:
            self._get_proj_name_case_name(query)

        return self._get_executable_commands(sim, query, file_location)

    def _get_executable_commands(self, sim, query, file_location):
        """Returns list of ExecutableCommand objects for a run.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with SMS.
            file_location (:obj:`str`): The location of input files for the simulation.

        Returns:
            See description.

        """
        self._exes = _get_sms_srhw_exes()
        self._file_location = file_location
        self._clear_output_folder(file_location)
        self._hydro_file = os.path.join(file_location, self.proj_name + '.srhwhydro')
        self._model_order = 1
        # If the hydro file does not exist, give up and tell the user to quit being dumb.
        if not os.path.isfile(self._hydro_file):
            XmEnv.report_error(
                f'Unable to run SRH-W because "{self._hydro_file}" was not found. Ensure the simulation has been '
                'exported.'
            )
            return []

        pre = self._srh_pre_model()
        main = self._srh_model()
        post = self._srh_post_model()

        runs = [pre, main]
        if post:
            runs.append(post)
        # read solution
        load_sol = self.get_solution_load_actions(sim, query, file_location)
        runs[-1].add_solution_file(load_sol[0])
        return runs

    def _srh_post_model(self):
        """Return the SRH Post model.

        Returns:
            (:obj:`ExecutableCommand`): see above
        """
        if self.query_helper.using_ugrid:
            return None

        cmd = os.path.join(os.path.dirname(__file__), 'srh_post.py')
        post = ExecutableCommand(executable=cmd, model='SRH-W', executable_order=self._model_order,
                                 display_name='PostSRH-W', run_weight=10, executable_is_script=True)
        post.executable_is_script = True
        post.set_run_group_and_id(0, 0)
        self._model_order += 1
        post.add_commandline_arg(f'-j {os.path.join(self._file_location, "srh_post.json")}')
        return post

    def _srh_model(self):
        """Return the SRH model.

        Returns:
            (:obj:`ExecutableCommand`): see above
        """
        main = ExecutableCommand(executable='SRH-W', model='SRH-W', executable_order=self._model_order,
                                 display_name='SRH-W', run_weight=75, plot_group='progressGroup')
        main.set_run_group_and_id(0, 0)
        self._model_order += 1
        main.add_commandline_arg(f'{self.proj_name}')
        return main

    def _srh_pre_model(self):
        """Return the SRH Pre model.

        Returns:
            (:obj:`ExecutableCommand`): see above
        """
        pre = ExecutableCommand(executable='PreSRH-W', model='SRH-W', executable_order=self._model_order,
                                display_name='PreSRH-W', run_weight=2)
        pre.set_run_group_and_id(0, 0)
        self._model_order += 1
        pre.add_commandline_arg('part')
        pre.add_commandline_arg('2')
        pre.add_commandline_arg(f'{self.proj_name}')
        return pre

    def _get_proj_name_case_name(self, query):
        """Does several things.

        Args:
            query:

        Returns:
            sim_comp (:obj:`SimComponent`): The SimComponent.
        """
        self.proj_name = os.path.splitext(os.path.basename(query.xms_project_path))[0]
        sim_comp = self.query_helper.sim_component

        # Set case name
        self.case_name = sim_comp.data.hydro.case_name
        # set scenario names if they exist
        return sim_comp

    @staticmethod
    def _clear_output_folder(filelocation):
        """Clears the output folder in anticipation of a model run.

        Args:
            filelocation (:obj:`str`): The location of input files for the simulation.

        """
        skip_files = ['_sif.dat', '_sof.dat', '_erosion.dat', '_radar_att.dat', '_rainfall.dat',
                      '_spatial_groundwater.dat', '_land_soil.dat', '_land_use.dat', '_soil_type.dat']
        for _, _, files in os.walk(filelocation):
            for f in files:
                if f.lower().endswith(tuple(skip_files)):
                    continue
                _, ext = os.path.splitext(f)
                if ext.lower() in ['.dat', '.h5']:
                    filesystem.removefile(os.path.join(filelocation, f))

        out_misc_dirs = [str(f.absolute()) for f in Path(filelocation).rglob('Output_MISC')]
        for out_misc in out_misc_dirs:
            if os.path.isdir(out_misc):
                shutil.rmtree(out_misc, ignore_errors=True)

    def read_solution(self, query, params, win_cont):
        """Reads the SRH Solution.

        Args:
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with GMS.
            params (:obj:`dict`): Generic map of parameters. Contains the structures for various components that
             are required for adding vertices to the Query Context with Add().
            win_cont (:obj:`QWidget`): Parent window
        """
        ds_file_name = params[0]['solution_file']
        rst_files = glob.glob(f'{os.path.dirname(ds_file_name)}{os.path.sep}*_RST*.dat')
        if rst_files:
            rst_times = [os.path.getmtime(rst) for rst in rst_files]
            max_time = max(rst_times)
            for rst_time, rst_file in zip(rst_times, rst_files):
                if rst_time < max_time:
                    os.remove(rst_file)

        scenario = ''
        if os.path.basename(os.path.dirname(os.path.dirname(ds_file_name))) != 'SRH-W':
            scenario = os.path.basename(os.path.dirname(ds_file_name))

        data_loc = 'NODE'
        if ds_file_name.endswith('XMDFC.h5'):
            data_loc = 'CELL'

        using_folders = False
        dataset_kwargs = []
        solution_file = h5py.File(ds_file_name, 'r')
        dataset_groups = solution_file.keys()
        if 'Datasets' in solution_file:
            dataset_groups = solution_file['Datasets'].keys()
        for ds in dataset_groups:
            if ds.lower() in ['guid', 'file version', 'file type']:
                continue
            if isinstance(ds, bytes):
                # There are some folders in the XMDF file that are bogus and include bytes in the name.  Skip these.
                continue
            ds_name = ds
            if 'Datasets' in solution_file:
                ds_name = f'Datasets/{ds}'
            dataset = Dataset(ds_file_name, ds_name, data_loc)

            dset_kwargs = {'do_dataset': dataset}
            if scenario:
                dset_kwargs['folder_path'] = scenario
                using_folders = True
            if ds.startswith('Max_') or ds.startswith('time_of_Max_'):
                path = ''
                if 'folder_path' in dset_kwargs:
                    path = dset_kwargs['folder_path']
                    path = path + '/'
                path = path + 'Max'
                if ds.startswith('time_of_Max_'):
                    path = path + '/time_of_max'
                dset_kwargs['folder_path'] = path
                using_folders = True
            dataset_kwargs.append(dset_kwargs)

        # This is just so the datasets show up in a consistent order in the tree.
        if using_folders:
            dataset_kwargs.reverse()

        for dset in dataset_kwargs:
            query.add_dataset(**dset)
        return [], []

    def get_solution_load_actions(self, sim, query, filelocation, all_scenarios=True):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with GMS.
            filelocation (:obj:`str`): The location of input files for the simulation.
            all_scenarios (:obj:`bool`): indicates if we should get the files for all scenarios

        Returns:
            (:obj:`listp[xms.api.dmi.ExecutableCommand]`):
                The executable objects to run and the action requests that go with it.
        """
        self._setup_query_helper(query)
        if not self.proj_name:
            self._get_proj_name_case_name(query)

        ug_sol = ''
        if self.query_helper.using_ugrid:
            ug_sol = 'C'
        file_name = os.path.join(filelocation, f'{self.proj_name}_XMDF{ug_sol}.h5')
        load_sol = ActionRequest(modality='MODAL', class_name='RunSRHW', module_name='xms.srhw.srhw_run',
                                 method_name='read_solution', parameters={'solution_file': file_name})
        return [load_sol]
