"""Read files used to store STWAVE simualtions as part of old SMS projects."""

# 1. Standard Python modules
import os
import traceback
import uuid

# 2. Third party modules
import numpy as np
import pandas

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.components.bases.migrate_base import MigrateBase
from xms.components.runners import migrate_runner as mgrun
from xms.data_objects.parameters import Component, Coverage, FilterLocation, julian_to_datetime, Simulation
from xms.guipy.time_format import ISO_DATETIME_FORMAT

# 4. Local modules
from xms.stwave.components.simulation_component import SimulationComponent
from xms.stwave.data import simulation_data


class XmsReaderDbVersion(MigrateBase):
    """A class for reading STWAVE simultions from files that are part of old SMS projects."""

    def __init__(self):
        """Constructor."""
        super().__init__()
        self._xms_data = None  # Old data to migrate. Get from Query or pass in for testing.
        self._old_takes = None  # passed in to first pass of xmscomponents migrate process
        self._old_items = None  # passed in to first pass of xmscomponents migrate process
        self._widgets = None  # passed in to first pass of xmscomponents migrate process
        self._errors = []
        self._query = None

        # Stuff we want to send back to SMS
        self._delete_uuids = []  # Things we want to delete: [uuid]
        self._new_sims = []  # [(do_simulation, do_component)]
        self._take_uuids = []  # [(taken_uuid, taker_uuid)]
        self._new_covs = []  # new coverages for monitor and nesting

    def _get_coverages_to_migrate(self, query):
        """Get dumps of all the coverages we need to migrate.

        Args:
            query (xms.dmi.Query): Object for communicating with SMS.
        """
        for item_uuid, info in self._old_items.items():
            # info = (modelname, typename, entitytype, simid)
            if info[0] == 'STWAVE' and info[2] == 'Coverage':
                cov_dump = query.item_with_uuid(item_uuid)
                if not cov_dump:
                    continue
                self._xms_data['cov_dumps'][item_uuid] = (cov_dump, info[1])
                self._xms_data['projection'] = cov_dump.projection

    def _get_xms_data(self, query):
        """Get the XMS temp component directory for creating new components.

        self._xms_data = {
            'comp_dir': ''  # XMS temp component directory
            'cov_dumps': {}  # Old coverages to migrate {uuid: (dump object, coveragetype)}
            'projection': xms.data_objects.parameters.Projection  # Projection of exported items (should be same)
            'proj_dir': str  # Path to the project's folder
        }

        Args:
            query (xms.dmi.Query): Object for communicating with SMS.
        """
        self._query = query
        if self._xms_data:  # Initial XMS data provided (probably testing)
            return

        self._xms_data = {
            'comp_dir': '',
            'cov_dumps': {},
            'projection': None,
            'proj_dir': '',
            'map_tree_root': None,
        }
        try:
            self._xms_data['proj_dir'] = os.path.dirname(mgrun.project_db_file)
            # Get the SMS temp directory
            self._xms_data['comp_dir'] = os.path.join(os.path.dirname(query.xms_temp_directory), 'Components')
            self._get_coverages_to_migrate(query)
            # Get the Map Data tree root
            self._xms_data['map_tree_root'] = tree_util.first_descendant_with_name(query.project_tree, 'Map Data')
        except Exception as ex:
            traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
            self._errors.append(('ERROR', f'Could not retrieve SMS data required for migration:\n"{traceback_msg}'))

    def _migrate_model_control(self, old_sim_uuid):
        """Migrate the simulation Model Control widgets.

        Args:
            old_sim_uuid (str): UUID of the old simulation

        Returns:
            xms.data_objects.parameters.Component, ModelControl, str: The new simulation's hidden component, its data,
                and path to the main file where data should be written.
        """
        # Create a folder and UUID for the new sim component.
        comp_uuid = self._get_next_sim_comp_uuid()
        sim_comp_dir = os.path.join(self._xms_data['comp_dir'], comp_uuid)
        os.makedirs(sim_comp_dir, exist_ok=True)

        # Create the component data_object to send back to SMS.
        sim_main_file = os.path.join(sim_comp_dir, 'simulation_comp.nc')
        sim_comp = Component(main_file=sim_main_file, model_name='STWAVE', unique_name='Sim_Component',
                             comp_uuid=comp_uuid)

        # Fill component data from widget values in the old database.
        sim_py_comp = SimulationComponent(sim_main_file)  # Initialize some default data
        data = sim_py_comp.data
        try:
            if old_sim_uuid in self._widgets:
                sim_widgets = self._widgets[old_sim_uuid]['Simulation'][-1]

                # General options
                data.info.attrs['plane'] = sim_widgets['cbxPlane'][0][2]
                data.info.attrs['current_interaction'] = sim_widgets['cbxCurr'][0][2]
                data.info.attrs['breaking_type'] = sim_widgets['cbxBreakType'][0][2]
                data.info.attrs['source_terms'] = sim_widgets['cbxTerms'][0][2]
                data.info.attrs['rad_stress'] = sim_widgets['chkRadStress'][0][2]
                data.info.attrs['output_stations'] = sim_widgets['chkOutputStations'][0][2]
                data.info.attrs['interpolation'] = sim_widgets['cbxInterp'][0][2]
                data.info.attrs['c2shore'] = sim_widgets['chkC2Shore'][0][2]
                data.info.attrs['friction'] = sim_widgets['cbxFric'][0][2]
                data.info.attrs['JONSWAP'] = sim_widgets['edtJONSWAP'][0][2]
                data.info.attrs['manning'] = sim_widgets['edtManning'][0][2]
                data.info.attrs['depth'] = sim_widgets['cbxDepth'][0][2]
                data.info.attrs['surge'] = sim_widgets['cbxSurge'][0][2]
                data.info.attrs['wind'] = sim_widgets['cbxWind'][0][2]
                data.info.attrs['ice'] = sim_widgets['cbxIce'][0][2]
                data.info.attrs['ice_threshold'] = sim_widgets['edtIceThresh'][0][2]
                data.info.attrs['side1'] = sim_widgets['cbxSide1'][0][2]
                data.info.attrs['side2'] = sim_widgets['cbxSide2'][0][2]
                data.info.attrs['side3'] = sim_widgets['cbxSide3'][0][2]
                data.info.attrs['side4'] = sim_widgets['cbxSide4'][0][2]
                data.info.attrs['location_coverage'] = sim_widgets['chkUseLocCov'][0][2]
                data.info.attrs['max_init_iters'] = sim_widgets['edtMaxInitIters'][0][2]
                data.info.attrs['init_iters_stop_value'] = sim_widgets['edtInitItersStopVal'][0][2]
                data.info.attrs['init_iters_stop_percent'] = sim_widgets['edtInitItersStopPercent'][0][2]
                data.info.attrs['max_final_iters'] = sim_widgets['edtMaxFinalIters'][0][2]
                data.info.attrs['final_iters_stop_value'] = sim_widgets['edtFinalItersStopVal'][0][2]
                data.info.attrs['final_iters_stop_percent'] = sim_widgets['edtFinalItersStopPercent'][0][2]
                if 'edtProcI' in sim_widgets:  # Was hidden in last DB interface version release
                    data.info.attrs['processors_i'] = sim_widgets['edtProcI'][0][2]
                if 'edtProcJ' in sim_widgets:  # Was hidden in last DB interface version release
                    data.info.attrs['processors_j'] = sim_widgets['edtProcJ'][0][2]
                data.info.attrs['min_frequency'] = sim_widgets['edtMinFreq'][0][2]
                data.info.attrs['delta_frequency'] = sim_widgets['edtDeltaFreq'][0][2]
                data.info.attrs['num_frequencies'] = sim_widgets['edtNumFreq'][0][2]
                data.info.attrs['boundary_source'] = sim_widgets['cbxBoundarySource'][0][2]
                data.info.attrs['angle_convention'] = sim_widgets['cbxAngConv'][0][2]

                data.info.attrs['depth_uuid'] = self._dset_val_from_grid(sim_widgets['dsetDep'][0][2],
                                                                         sim_widgets["dsetDep"][1][2])
                data.info.attrs['current_uuid'] = self._dset_val_from_grid(sim_widgets['dsetCurr'][0][2],
                                                                           sim_widgets["dsetCurr"][1][2])
                data.info.attrs['JONSWAP_uuid'] = self._dset_val_from_grid(sim_widgets['dsetJONSWAP'][0][2],
                                                                           sim_widgets["dsetJONSWAP"][1][2])
                data.info.attrs['manning_uuid'] = self._dset_val_from_grid(sim_widgets['dsetManning'][0][2],
                                                                           sim_widgets["dsetManning"][1][2])
                data.info.attrs['surge_uuid'] = self._dset_val_from_grid(sim_widgets['dsetSurge'][0][2],
                                                                         sim_widgets["dsetSurge"][1][2])
                data.info.attrs['wind_uuid'] = self._dset_val_from_grid(sim_widgets['dsetWind'][0][2],
                                                                        sim_widgets["dsetWind"][1][2])
                data.info.attrs['ice_uuid'] = self._dset_val_from_grid(sim_widgets['dsetIce'][0][2],
                                                                       sim_widgets["dsetIce"][1][2])
                data.info.attrs['location_coverage_uuid'] = sim_widgets['LocationCovSelector'][0][2] if \
                    sim_widgets['LocationCovSelector'][0][2] != 'cccccccc-cccc-cccc-cccc-cccccccccccc' else ''
                data.info.attrs['output_stations_uuid'] = sim_widgets['OutputStationCovSelector'][0][2] if \
                    sim_widgets['OutputStationCovSelector'][0][2] != 'cccccccc-cccc-cccc-cccc-cccccccccccc' else ''

                # Case times
                water_level = sim_widgets.get('tblColWaterLevel', [])
                wind_mag = sim_widgets.get('tblColWindMag', [])
                wind_dir = sim_widgets.get('tblColWindDir', [])
                times = sim_widgets.get('tblColTime', [])
                time_vals = []
                dir_vals = []
                mag_vals = []
                surge_vals = []
                reftime = None
                for level, mag, dir_val, time_val in zip(water_level, wind_mag, wind_dir, times):
                    dir_vals.append(float(dir_val[2]))
                    mag_vals.append(float(mag[2]))
                    surge_vals.append(float(level[2]))
                    # Old database stored datetimes as stringified Julian dates
                    dt = julian_to_datetime(float(time_val[2]))
                    if reftime is None:  # Use the first time as the reference date
                        reftime = dt
                        data.info.attrs['reftime'] = reftime.strftime(ISO_DATETIME_FORMAT)
                    time_vals.append((dt - reftime).total_seconds())

                # Convert second offsets to default units of hours and throw away garbage precision
                time_vals = np.array(time_vals, dtype=np.float64) / 3600.0  # Convert to default units of hours
                time_vals = np.around(time_vals, 3)  # Throw away excess precision
                case_time_data = simulation_data.case_data_table(time_vals, dir_vals, mag_vals, surge_vals)
                data.case_times = pandas.DataFrame(case_time_data).to_xarray()
        except Exception as ex:
            traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
            self._errors.append(('ERROR', f'Could not migrate STWAVE model control:\n"{traceback_msg}'))

        return sim_comp, data, sim_main_file

    def _get_next_sim_comp_uuid(self):
        """Get a randomly generated UUID for a new sim component or hard-coded one for testing."""
        if 'new_sim_comp_uuids' in self._xms_data and self._xms_data['new_sim_comp_uuids']:
            return self._xms_data['new_sim_comp_uuids'].pop(0)
        return str(uuid.uuid4())

    def _dset_val_from_grid(self, grid_uuid, path):
        """Finds the value to set in the attrs for one of the datasets, based on info read from widgets."""
        if grid_uuid == 'cccccccc-cccc-cccc-cccc-cccccccccccc':
            return ''
        node = tree_util.find_tree_node_by_uuid(tree_node=self._query.project_tree, uuid=grid_uuid)
        if not node:
            return ''
        tree_item = tree_util.item_from_path(tree=node, path=os.path.join(node.name, path))
        if tree_item is None:
            return ''
        return tree_item.uuid

    def _migrate_sims(self):
        """Migrate simulations to current component based version.

        Returns:
            dict: Mapping of old simulation to list containing the UUID of the new simulation that replaces it.
                Needed for second pass of migration.
        """
        replace_map = {}
        for item_uuid, info in self._old_items.items():
            # Look for STWAVE simulations. Filter out non-visible simulations. They are garbage. Tree item names
            # cannot contain '*' characters.
            # info = (modelname, simname, entitytype, simid)
            if info[0] == 'STWAVE' and info[2] == 'Simulation' and not info[1].startswith('*'):
                try:
                    # Create a new simulation and its hidden component
                    new_sim_uuid = str(uuid.uuid4())
                    new_sim = Simulation(model='STWAVE', sim_uuid=new_sim_uuid, name=info[1])
                    replace_map[item_uuid] = [new_sim_uuid]
                    sim_comp, sim_data, main_file = self._migrate_model_control(item_uuid)

                    # Add the new simulation and its component to the Query.
                    self._new_sims.append((new_sim, sim_comp))

                    # Search for a taken items under the old simulation.
                    if item_uuid in self._old_takes:
                        for take_uuid in self._old_takes[item_uuid]:
                            if take_uuid not in self._old_items:
                                continue

                            type_name = self._old_items[take_uuid][1]

                            if type_name == 'CGRID':
                                # No migration needed for the grid, just relink.
                                sim_data.info.attrs['grid_uuid'] = take_uuid
                                self._take_uuids.append((take_uuid, new_sim_uuid))
                            elif type_name == 'Monitoring Cells':  # 0 or 1 per simulation
                                # Create a new monitoring coverage, and copy over the points from the old
                                # existing coverage.  Store the uuid's so we can add, link, and delete later.
                                old_cov = self._xms_data['cov_dumps'][take_uuid][0]
                                uuid_val = str(uuid.uuid4())
                                new_cov = Coverage(uuid=uuid_val, name=old_cov.name)
                                points = old_cov.get_points(FilterLocation.LOC_DISJOINT)
                                new_cov.set_points(points)
                                new_cov.complete()
                                sim_data.info.attrs['monitoring'] = 1
                                sim_data.info.attrs['monitoring_uuid'] = uuid_val
                                self._new_covs.append((new_cov, take_uuid))  # Store the new cov
                                self._delete_uuids.append(take_uuid)  # Add the old coverage uuid to delete list
                            elif type_name == 'Nesting Points':  # 0 or 1 per simulation
                                # Create a new monitoring coverage, and copy over the points from the old
                                # existing coverage.  Store the uuid's so we can add, link, and delete later.
                                old_cov = self._xms_data['cov_dumps'][take_uuid][0]
                                uuid_val = str(uuid.uuid4())
                                new_cov = Coverage(uuid=uuid_val, name=old_cov.name)
                                points = old_cov.get_points(FilterLocation.LOC_DISJOINT)
                                new_cov.set_points(points)
                                new_cov.complete()
                                sim_data.info.attrs['nesting'] = 1
                                sim_data.info.attrs['nesting_uuid'] = uuid_val
                                self._new_covs.append((new_cov, take_uuid))  # Store the new cov
                                self._delete_uuids.append(take_uuid)  # Add the old coverage uuid to delete list
                            elif type_name == 'MAP':  # 0 or 1 per simulation
                                # No migration needed for the Spectral coverage, just relink.
                                sim_data.info.attrs['spectral_uuid'] = take_uuid
                                self._take_uuids.append((take_uuid, new_sim_uuid))

                    # Write sim component data to main file.
                    sim_data.commit()
                except Exception as ex:
                    traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
                    self._errors.append(('ERROR', f'Could not migrate STWAVE simulation:\n"{traceback_msg}'))

        return replace_map

    def _build_coverage_tree_path(self, old_uuid):
        """Build the folder path a coverage should be placed into, if the old coverage was inside a folder.

        Args:
            old_uuid (str): UUID of the old coverage

        Returns:
            str: The folder path the new coverage should have under its parent, or empty string if the old coverage
                was not in a folder.
        """
        map_root = self._xms_data.get('map_tree_root')
        tree_path = tree_util.build_tree_path(map_root, old_uuid)
        split_path = tree_path.split('/', 1)  # Remove 'Map Data' from the folder tree path
        if len(split_path) > 1:
            return os.path.dirname(split_path[1])  # Remove the coverage name from the folder tree path
        return ''

    def map_to_typename(self, uuid_map, widget_map, take_map, main_file_map, hidden_component_map,
                        material_atts, material_polys, query):
        """This is the first pass of a two-pass system.

        This function performs all the data object migrations necessary for use between versions of model interfaces.

        Args:
            uuid_map (:obj:`dict` of :obj:`uuid` to :obj:): This is a map from uuid's to object attributes.
                This is a collection of things you want to find the conversions for.
            widget_map (:obj:`dict` of :obj:`uuid` to :obj:): This is a larger dictionary mapping uuid's to
                their conversions.
            take_map (:obj:`dict` of :obj:`uuid` to :obj:): This is a map from uuid's of taking
                objects to a list of uuids of objects taken by the taker.
            main_file_map (:obj:`dict` of :obj:`uuid` to str): This is a map from uuid's to main
                files of components.
            hidden_component_map (:obj:`dict` of :obj:`uuid` to :obj:): This is a map from uuid's of owning
                objects to a list of uuids of hidden components owned by the object.
            material_atts (:obj:`dict`): Dictionary whose key is coverage UUID and value is dict whose key is
                material id and value is tuple of display options (name, r, g, b, alpha, texture)
            material_polys (:obj:`dict`): Dictionary whose key is coverage UUID and value is dict whose key is
                material id and value is a set of polygon ids with that material assignment.
            query (:obj:`xms.api.dmi.Query`): This is used to communicate with XMS.
                Do not call the 'send' method on this instance as 'send' will be called later.

        Returns:
            (:obj:`dict` of :obj:`uuid` to :obj:): This is the map of delete and replacement requests.
        """
        self._old_items = uuid_map
        self._old_takes = take_map
        self._widgets = widget_map
        if query:
            self._get_xms_data(query)

        replace_map = {}
        try:
            # Migrate all STWAVE simulations
            replace_map.update(self._migrate_sims())
        except Exception as ex:
            traceback_msg = ''.join(traceback.format_exception(type(ex), ex, ex.__traceback__))
            self._errors.append((
                'ERROR', f'Could not migrate STWAVE project:\n"{traceback_msg}'
            ))
            return {}

        return replace_map

    def send_replace_map(self, replace_map, query):
        """This is the second pass of a two-pass system.

        Args:
            replace_map (:obj:`dict` of :obj:`uuid` to :obj:): This is the map of delete and replace requests.
                This can be generated by map_to_typename().
            query (:obj:`xms.api.dmi.Query`): This is used to communicate with SMS.
        """
        # Add new simulations to the Query
        for new_sim in self._new_sims:
            query.add_simulation(new_sim[0], components=[new_sim[1]])

        # Add new monitoring and nesting coverages, and delete the old ones
        for cov in self._new_covs:
            query.add_coverage(cov[0], folder_path=self._build_coverage_tree_path(cov[1]))
        for delete_uuid in self._delete_uuids:
            query.delete_item(delete_uuid)

        # Link up takes to the new simulations
        for take_pair in self._take_uuids:
            query.link_item(taker_uuid=take_pair[1], taken_uuid=take_pair[0])

    def get_messages_and_actions(self):
        """Called at the end of migration.

        This is for when a message needs to be given to the user about some change due to migration.
        Also, an ActionRequest can be given if there is ambiguity in the migration.

        Returns:
            (list of tuple of str, list of xms.api.dmi.ActionRequest):
                messages, action_requests - Where messages is a list of tuples with the first element of the tuple being
                the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.
                action_requests is a list of actions for XMS to perform.
        """
        if self._errors:
            messages = [('ERROR', 'Error(s) encountered during migration of STWAVE project.')]
            messages.extend(self._errors)
        else:
            messages = [('INFO', 'Successfully migrated STWAVE project.')]

        return messages, []
