"""Creates a coverage with polygons."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core.coverage_builder import CoverageBuilder

# 4. Local modules


class PolygonCoverageBuilder:
    """Given points and lists of lines defining polygons (perhaps with holes), creates a coverage with the polygons.

    - output is a GeoDataFrame Coverage
    - build_coverage: Builds the GeoDataFrame Coverage.

    Glossary:
        I've tried to be consistent with the following terms to help avoid
        confusion:

        poly = A poly is a list of point indexes with the last point
        equal to the first point. Poly points can be in any order, clockwise
        or counterclockwise: [9, 3, 1, 6, 9]

        polygon = A polygon is a list of poly, outer first, then inner holes
        if any: [[3, 6, 8, 2, 3], [6, 7, 8, 6]]

        multipoly = list of polygon, each one associated with the same dataset value.
    """
    def __init__(self, point_locations, wkt, coverage_name, logger=None):
        """Initializes the class.

        Args:
            point_locations: List of xyz points which are 3-tuples, e.g. [(1.0, 3.0, 0.0), (2.0, 1.0, 1.0)]
            wkt: The map projection's wkt.
            coverage_name (str): Name to be given to the new coverage.
            logger (Optional[Logger]): The logger to use. If not provided will be the xms.tool logger.
        """
        self._wkt = wkt
        self._coverage_name = coverage_name
        self._logger = logger if logger is not None else logging.getLogger('xms.tool')

        self._pt_locs = point_locations  # xyz locations of points

        # Stuff for building a coverage
        self._pt_hash = {}  # Dict of point idx -> Point
        self._arc_hash = {}  # Dict of (node id1, node id2) -> Arc
        # For building the GeoDataFrame
        crs = ''
        if self._wkt is not None:
            crs = self._wkt
        self._coverage_builder = CoverageBuilder(cov_wkt=crs, cov_name=coverage_name)

    def _get_or_hash_point(self, point_idx: int) -> int:
        """Get an existing point using an int hash (creates the node if it doesn't exist).

        Args:
            point_idx (int): Grid point index.

        Returns:
            int: The index of the existing or newly created node associated with the passed in location
        """
        if point_idx not in self._pt_hash:
            node_id = self._coverage_builder.get_node((self._pt_locs[point_idx][0], self._pt_locs[point_idx][1],
                                                      self._pt_locs[point_idx][2]))
            self._pt_hash[point_idx] = node_id
        return self._pt_hash[point_idx]

    def _get_or_hash_arc(self, start_node: int, end_node: int) -> int:
        """Get an existing arc ID using end node ids (creates the arc if it doesn't exist).

        Order of the end nodes does not matter. Arcs are hashed by the ordered pair of their end node ids.

        Args:
            start_node (int): The list index of the arc's start node
            end_node (int): The list index of the arc's end node

        Returns:
            int: The ID of the existing or newly created arc associated with the passed in end node ids
        """
        node1_id = start_node
        node2_id = end_node
        if node1_id < node2_id:
            arc_hash = hash((node1_id, node2_id))
        else:
            arc_hash = hash((node2_id, node1_id))
        if arc_hash not in self._arc_hash:
            pt1 = self._coverage_builder.get_node_geometry(start_node)
            pt2 = self._coverage_builder.get_node_geometry(end_node)
            self._arc_hash[arc_hash] = self._coverage_builder.add_arc([pt1.coords[0], pt2.coords[0]])
        return self._arc_hash[arc_hash]

    def _build_polygon_arcs(self, poly, arcs, hash_arcs):
        """Construct a polygon arc definition given its points.

        Args:
            poly (iterable): List of polygon point definitions, in order
            arcs (list): Out variable to return the arc IDs that define the polygon
            hash_arcs (bool): If True, will hash arcs to eliminate duplicates.
        """
        # Get the arcs for the outer polygon
        # x, y = poly.exterior.coords.xy
        # z = [coord[-1] for coord in poly.exterior.coords[:-1]]
        if len(poly) > 3:
            if hash_arcs:
                pfirst = self._get_or_hash_point(poly[0])
                p1 = None
                p0 = pfirst
                for i in range(1, len(poly) - 1):
                    p1 = self._get_or_hash_point(poly[i])
                    arc = self._get_or_hash_arc(p0, p1)
                    arcs.append(arc)
                    p0 = p1

                arc = self._get_or_hash_arc(p1, pfirst)
            else:  # Create one arc, not worrying about combining duplicate features.
                node_locs = [tuple(self._pt_locs[node_idx]) for node_idx in poly]
                arc = self._coverage_builder.add_arc(node_locs)
            arcs.append(arc)

    def build_coverage(self, multipolys, hash_arcs):
        """Creates a GeoDataFrame Coverage with the Polygons and Arcs.

        Args:
            multipolys: Dict of dataset value -> multipoly. A multipoly
             is a list of polygon and a polygon is a list of poly (outer, inners),
             and a poly is a list of ugrid point indexes with the last point
             equal to the first point.
            hash_arcs (bool): If True, will hash arcs to eliminate duplicates.

        Returns:
            GeoDataFrame Coverage.

        """
        self._logger.info('Creating coverage polygons...')
        for multi_poly in multipolys.values():
            for polygons in multi_poly:

                # outer polygon is first
                outer_arcs = []
                self._build_polygon_arcs(polygons[0], outer_arcs, hash_arcs)

                # inner polygons follow
                inner_polys = []
                for poly in polygons[1:]:
                    inner_arcs = []
                    self._build_polygon_arcs(poly, inner_arcs, hash_arcs)
                    inner_polys.append(inner_arcs)

                # Create a Polygon
                polygon_arc_ids = []
                points = []
                for arc_id in outer_arcs:
                    points.append(list(self._coverage_builder.get_arc_geometry(arc_id).coords))
                    polygon_arc_ids.append(arc_id)
                interior_arc_ids = []
                holes = []
                for inner_poly in inner_polys:
                    interior_poly_arc_ids = []
                    inner_ring = []
                    for arc_id in inner_poly:
                        inner_ring.append(list(self._coverage_builder.get_arc_geometry(arc_id).coords))
                        interior_poly_arc_ids.append(arc_id)
                    holes.append(inner_ring)
                    interior_arc_ids.append(interior_poly_arc_ids)
                self._coverage_builder.add_polygon(points, holes, polygon_arc_ids, interior_arc_ids)
        # Create the coverage
        self._logger.info('Writing coverage to disk...')
        return self._coverage_builder.build_coverage()
