"""UGridFromCoverageTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass
import itertools
import math
import time
from typing import List

# 2. Third party modules
import numpy as np
from pyproj.enums import WktVersion
from rtree import index
from shapely.validation import explain_validity

# 3. Aquaveo modules
from xms.constraint import UGridBuilder
from xms.gdal.rasters import raster_utils as ru
from xms.gdal.utilities import gdal_utils as gu
from xms.grid.geometry.geometry import angle_between_edges_2d, point_in_polygon_2d
from xms.grid.ugrid import UGrid as XmUGrid
from xms.interp.interpolate import InterpIdw
from xms.interp.interpolate.interp_linear import InterpLinear
from xms.interp.interpolate.interp_linear_extrap_idw import InterpLinearExtrapIdw
import xms.mesher
from xms.mesher import meshing
from xms.mesher.meshing.refine_point import RefinePoint
from xms.tool_core import Argument, Tool

# 4. Local modules
from xms.tool.algorithms.coverage.grid_cell_to_polygon_coverage_builder import GridCellToPolygonCoverageBuilder
from xms.tool.algorithms.geometry.geometry import _is_inside_sm, run_parallel_points_in_polygon
from xms.tool.algorithms.ugrids import UGrid2dMerger
from xms.tool.algorithms.ugrids.ugrid_2d_merger import xy_distance
from xms.tool.utilities.coverage_conversion import get_polygon_point_lists

ARG_INPUT_COVERAGE = 0
ARG_INPUT_MERGE_TRIS = 1
ARG_INPUT_MERGE_TRIS_ANGLE = 2
ARG_INPUT_MESH_NAME = 3


class UGridFromCoverageTool(Tool):
    """Create a mesh from a coverage."""

    # MeshPolyEnum from XMS
    MESH_NONE = 0
    MESH_PATCH = 1
    MESH_PAVE = 2
    MESH_SPAVE = 3
    MESH_CPAVE = 4
    MESH_UGRID_BRIDGE = 5
    MESH_UGRID_PATCH = 6
    MESH_END = 7

    # bathpolytype enum from XMS
    BATH_CONSTANT = 0
    BATH_SCAT = 1
    BATH_RASTER = 2
    BATH_EXIST = 3
    BATH_FILL_POLY_FROM_EDGES = 4

    # Node merge option enum from XMS
    NODESPLIT = 0
    NODEMERGE = 1
    NODEDEGEN = 2

    # interptype enum from XMS
    LINEAR = 0
    IDW = 1
    NN = 2

    # extraptype enum from XMS
    IDWEXTRAP = 0
    SINGLEVALUE = 1

    # weightsourcetype enum from XMS
    NPOINTS = 0
    NPOINTSINQUAD = 1
    ALLPOINTS = 2

    @dataclass(frozen=True)
    class InterpKey:
        """This struct holds the settings for an individual interpolator."""
        geom_uuid: str
        dset_uuid: str
        bathy_interp_opt: int = -1
        bathy_extrap_opt: int = -1
        idw_num: int = -1
        extrap_val: float = -1.0
        truncate_min: float = -1.0
        truncate_max: float = -1.0

    def __init__(self, name='UGrid from Coverage'):
        """Initializes the class."""
        super().__init__(name=name)

        # Mesh coverage properties
        self._poly_ids = []
        self._mesh_type = []  # MESH_* constants
        self._bathy_type = []  # BATH_* constants
        self._bias = []
        self._cell_size = []
        self._const_elev = []
        self._interp_option_spave = []  # interptype constants
        self._extrap_option_spave = []  # extraptype constants
        self._idw_use_spave = []  # weightsourcetype constants for interp option
        self._idw_num_use_spave = []  # for interp
        self._truncate_spave = []
        self._truncate_min_spave = []
        self._truncate_max_spave = []
        self._timestep_idx = []
        self._extrap_const_spave = []
        self._geom_uuid_spave = []
        self._dset_uuid_spave = []
        self._interp_option_bathy = []  # interptype constants
        self._extrap_option_bathy = []  # extraptype constants
        self._idw_use_bathy = []  # weightsourcetype constants
        self._idw_num_use_bathy = []
        self._truncate_bathy = []
        self._truncate_min_bathy = []
        self._truncate_max_bathy = []
        self._timestep_idx_bathy = []
        self._extrap_const_bathy = []
        self._geom_uuid_bathy = []
        self._dset_uuid_bathy = []
        self._raster_uuid = []
        self._ugrid_bridge_uuid = []
        self._points_list = None
        self._merge_triangles = False

        # patching nodes
        self._polygon_node_merge = {}

        # Refine point attributes
        self._point_ids = []
        self._refine_on = []
        self._create_mesh_node = []
        self._element_size = []
        self._merge_option = []  # NODE* constants
        self._all_disjoint_pts = None  # Refine point DataFrame
        self._all_disjoint_locs = None  # Refine point DataFrame
        self._refine_pts = []

        self._mesh_cov = None
        self._disjoint_arcs = {}
        self._raster_to_poly_input = {}
        self._poly_count = 0
        self._poly_nodes_visited = set()
        self._poly_corner_locs = []

        self._ugrids_to_merge = []

        self._ugrid_patch_polys = []
        self._small_angle_to_id = {}
        self._cell_id_to_long_edge = {}

        self._cell_id_to_min_angle_info = {}

        self._new_cellstreams = {}
        self._id_to_new_cellstream = {}

        self._force_ugrid = True
        self._geom_txt = 'UGrid'

        self._poly_cov = None

        # Cache stuff
        self._input_scatters = {}
        self._input_scatter_points = {}
        self._input_scatter_cells = {}
        self._input_scatter_interpers = {}

        self.start_time = time.time()
        self.log_interval = 3.0  # Report xmsmesher log messages a maximum of once per 8 seconds

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.coverage_argument(name='input_coverage', description='Coverage',
                                   optional=False),
            self.bool_argument(name='merge_tris', description='Merge triangles', optional=False, value=False),
            self.float_argument(name='merge_angle', description='Merge triangles minimum angle', optional=True,
                                value='65.0', hide=True),
            self.string_argument(name='mesh_name', description=f'Name of the output {self._geom_txt}', value='',
                                 optional=True),
        ]
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}

        if arguments[ARG_INPUT_MERGE_TRIS].value is True:
            angle = arguments[ARG_INPUT_MERGE_TRIS_ANGLE].value
            if angle < 0.0 or angle > 90.0:
                errors[arguments[ARG_INPUT_MERGE_TRIS_ANGLE].name] = 'Minimum merge angle must be between 0.0 and 90.0.'

        return errors

    def enable_arguments(self, arguments: List[Argument]):
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments(list): The tool arguments.
        """
        arguments[ARG_INPUT_MERGE_TRIS_ANGLE].show = arguments[ARG_INPUT_MERGE_TRIS].value is True

    def _get_poly_atts(self, atts):
        """Get the polygon attributes."""
        if 'Id' in atts:
            self._poly_ids = list[int](atts['Id'])
            self._mesh_type = list[int](atts['MeshingType'])
            self._bathy_type = list[int](atts['BathyType'])
            self._bias = list[float](atts['Bias'])
            self._cell_size = list[float](atts['Cellsize'])
            self._const_elev = list[float](atts['ConstantElev'])
            self._interp_option_spave = list[int](atts['SizeInterpOpt'])
            self._extrap_option_spave = list[int](atts['SizeExtrapOpt'])
            self._idw_use_spave = list[int](atts['SizeExtrapIdwUse'])
            self._idw_num_use_spave = list[int](atts['SizeExtrapIdwNum'])
            self._truncate_spave = list[int](atts['SizeTruncate'])
            self._truncate_min_spave = list[float](atts['SizeTruncateMin'])
            self._truncate_max_spave = list[float](atts['SizeTruncateMax'])
            self._timestep_idx = list[int](atts['SizeTimestepIdx'])
            self._extrap_const_spave = list[float](atts['SizeExtrapVal'])
            self._geom_uuid_spave = list[str](atts['SizeGeomUuid'])
            self._dset_uuid_spave = list[str](atts['SizeDsetUuid'])
            self._interp_option_bathy = list[int](atts['ElevInterpOpt'])
            self._extrap_option_bathy = list[int](atts['ElevExtrapOpt'])
            self._idw_use_bathy = list[int](atts['ElevExtrapIdwUse'])
            self._idw_num_use_bathy = list[int](atts['ElevExtrapIdwNum'])
            self._truncate_bathy = list[int](atts['ElevTruncate'])
            self._truncate_min_bathy = list[float](atts['ElevTruncateMin'])
            self._truncate_max_bathy = list[float](atts['ElevTruncateMax'])
            self._timestep_idx_bathy = list[int](atts['ElevTimestepIdx'])
            self._extrap_const_bathy = list[float](atts['ElevExtrapVal'])
            self._geom_uuid_bathy = list[str](atts['ElevGeomUuid'])
            self._dset_uuid_bathy = list[str](atts['ElevDsetUuid'])
            self._raster_uuid = list[str](atts['RasterUuid'])
            self._ugrid_bridge_uuid = list[str](atts['BridgeUGridUuid'])

    def _get_point_atts(self, atts):
        """Get the point attributes."""
        if 'PointId' not in atts:
            return
        self._point_ids = list[int](atts['PointId'])
        self._refine_on = list[bool](atts['RefineOn'])
        self._create_mesh_node = list[bool](atts['CreateMeshNode'])
        self._element_size = list[float](atts['ElementSize'])
        self._merge_option = list[int](atts['MergeOpt'])

    def get_mesh_cov_props(self):
        """Get the mesh coverage properties."""
        atts = self._mesh_cov.attrs['attributes']
        self._get_poly_atts(atts)

        # patching nodes
        self._read_poly_node_merge_flags(atts)

        # Point attributes
        self._get_point_atts(atts)
        self._all_disjoint_pts = self._mesh_cov[self._mesh_cov['geometry_types'] == 'Point']
        self._all_disjoint_locs = self._all_disjoint_pts.geometry.get_coordinates().to_numpy()

    def _read_poly_node_merge_flags(self, atts):
        """Read the polygon node merge option arrays into a map.

        Args:
            atts (dict): The mesh generation attributes
        """
        # Build the polygon node merge option map
        poly_ids = list[int](atts['PolyNodePolyId'])
        node_ids = list[int](atts['PolyNodeId'])
        merge_opts = list[int](atts['PolyNodeMergeOpt'])
        for poly_id, node_id, merge_opt in zip(poly_ids, node_ids, merge_opts):
            self._polygon_node_merge.setdefault(poly_id, {})[node_id] = merge_opt

    def _get_interpolator(self, idx, is_spave):
        """Gets the scatter interpolator for a given polygon's options.

        Args:
            idx (int): The meshing polygon index
            is_spave (bool): If True, returns the size function interpolator, otherwise the bathymetry elevation
                function interpolator.

        Returns:
            (Interpolator): See description
        """
        interp_opt = self._interp_option_spave[idx] if is_spave else self._interp_option_bathy[idx]
        extrap_opt = self._extrap_option_spave[idx] if is_spave else self._extrap_option_bathy[idx]
        geom_uuid, dset_uuid = self.get_scatter_info(idx, is_spave)

        # Get all the options for the interpolator to see if we have already constructed one.
        truncate_min = truncate_max = -1.0
        truncate = self._truncate_spave[idx] if is_spave else self._truncate_bathy[idx]
        if truncate == 1:
            truncate_min = self._truncate_min_spave[idx] if is_spave else self._truncate_min_bathy[idx]
            truncate_max = self._truncate_max_spave[idx] if is_spave else self._truncate_max_bathy[idx]
        idw_opt = self._idw_use_spave[idx] if is_spave else self._idw_use_bathy[idx]
        idw_num = -1
        if idw_opt != self.ALLPOINTS and (interp_opt == self.IDW or extrap_opt == self.IDWEXTRAP):
            idw_num = self._idw_num_use_spave[idx] if is_spave else self._idw_num_use_bathy[idx]
        extrap_val = -1.0
        if interp_opt != self.IDW and extrap_opt != self.IDWEXTRAP:
            extrap_val = self._extrap_const_spave[idx] if is_spave else self._extrap_const_bathy[idx]
        interp_key = self.InterpKey(geom_uuid=geom_uuid, dset_uuid=dset_uuid, bathy_interp_opt=interp_opt,
                                    bathy_extrap_opt=extrap_opt, idw_num=idw_num, extrap_val=extrap_val,
                                    truncate_min=truncate_min, truncate_max=truncate_max)
        interpolator = self._input_scatter_interpers.get(interp_key)

        if interpolator is None:
            points, scalars, triangles = self.get_scatter_data(geom_uuid, dset_uuid)
            use_quads = (idw_num == self.NPOINTSINQUAD)
            search_opt = 'natural_neighbor' if interp_opt == self.NN else 'nearest_points'
            self.logger.info('Building scatter set interpolator...')
            if interp_opt == self.IDW:
                interpolator = InterpIdw(points=points, triangles=triangles, scalars=scalars,
                                         number_nearest_points=idw_num, quadrant_oct=use_quads)
            else:

                if extrap_opt == self.IDWEXTRAP:
                    interpolator = InterpLinearExtrapIdw(points=points, triangles=triangles, scalars=scalars,
                                                         point_search_option=search_opt, quadrant_oct=use_quads,
                                                         number_nearest_points=idw_num)
                else:
                    interpolator = InterpLinear(points=points, scalars=scalars, triangles=triangles,
                                                point_search_option=search_opt)
                    interpolator.extrapolation_value = extrap_val
            if interpolator is not None and truncate == 1:
                interpolator.set_truncation(truncate_max, truncate_min)
            self._input_scatter_interpers[interp_key] = interpolator

        return interpolator

    def set_bathy_input(self, poly_input, idx):
        """Called to determine if arguments are valid.

        Args:
            poly_input (meshing.PolyInput): Input for meshing the polygon
            idx (int): polygon index for the coverage vectors

        Returns:
            (meshing.PolyInput): Input for meshing the polygon.
        """
        bathy_opt = self._bathy_type[idx]
        if bathy_opt == self.BATH_FILL_POLY_FROM_EDGES:
            return poly_input

        if bathy_opt == self.BATH_RASTER:
            # raster elevs will be handled after because that's how SMS does it
            raster = self._raster_uuid[idx]
            if raster in self._raster_to_poly_input.keys():
                self._raster_to_poly_input[raster].append(poly_input)
            else:
                list_input = [poly_input]
                self._raster_to_poly_input[raster] = list_input
            return poly_input

        interpolator = None
        if bathy_opt == self.BATH_CONSTANT:
            const_elev = self._const_elev[idx]
            points = [[0.0, 0.0, const_elev], [1.0, 1.0, const_elev], [0.0, 1.0, const_elev]]
            scalars = [const_elev, const_elev, const_elev]
            interpolator = InterpLinear(points=points, scalars=scalars)
            interpolator.extrapolation_value = const_elev
        elif bathy_opt == self.BATH_SCAT:
            interpolator = self._get_interpolator(idx, is_spave=False)

        poly_input.elevation_function = interpolator
        return poly_input

    def _get_input_ugrid(self, grid_name: str) -> XmUGrid:
        """Get/cache an input grid so we don't retrieve for every poly.

        Args:
            grid_name: The grid to retrieve.

        Returns:
            The constrained grid object.
        """
        ugrid = self._input_scatters.get(grid_name)
        if ugrid is None:
            ugrid = super().get_input_grid(grid_name).ugrid  # Only constuct the Python XmUGrid once
            self._input_scatters[grid_name] = ugrid
        return ugrid

    def _get_input_ugrid_points(self, grid_name: str, ugrid: XmUGrid) -> np.ndarray:
        """Get/cache an input grid's points so we don't have to retrieve for every poly.

        Args:
            grid_name: Tree item path of the grid.
            ugrid: The grid's geometry.

        Returns:
            The grid's location list.
        """
        points = self._input_scatter_points.get(grid_name)
        if points is None:
            points = ugrid.locations  # Only construct python point list once
            self._input_scatter_points[grid_name] = points
        return points

    def _get_input_ugrid_tris(self, grid_name: str, ugrid: XmUGrid) -> list:
        """Get/cache an input grid's tris so we don't have to retrieve for every poly.

        Notes:
            This only works for triangular geometries. I added this to optimize using large scatters as a bathy source.

        Args:
            grid_name: Tree item path of the grid.
            ugrid: The grid's geometry.

        Returns:
            The grid's tris as a list of 3-element tuples.
        """
        tris = self._input_scatter_cells.get(grid_name)
        if tris is None:
            cs = ugrid.cellstream
            tris = [cs[start + 2: start + 5] for start in range(0, len(cs), 5)]
            tris = list(itertools.chain(*tris))
            self._input_scatter_cells[grid_name] = tris
        return tris

    def get_scatter_info(self, idx, is_spave):
        """Get the UUIDs of an input scatter set and its elevation dataset.

        Args:
            idx (int): polygon index
            is_spave (bool): use spave data or bathy data

        Returns:
            (tuple[str, str]): Scatter UUID and dataset UUID
        """
        if is_spave:
            scatter_geom_uuid = self._geom_uuid_spave[idx]
            scatter_dset_uuid = self._dset_uuid_spave[idx]
        else:
            scatter_geom_uuid = self._geom_uuid_bathy[idx]
            scatter_dset_uuid = self._dset_uuid_bathy[idx]
        return scatter_geom_uuid, scatter_dset_uuid

    def get_scatter_data(self, geom_uuid, dset_uuid):
        """Called to determine if arguments are valid.

        Args:
            geom_uuid (str): UUID of the scatter set
            dset_uuid (str): UUID of the scatter set's elevation dataset

        Returns:
            points (list): The points of the source geometry as x,y,z tuples
            scalars (list): The scalar values at the points
            triangles (list): The triangles of the source geometry as point index tuples
        """
        points = []
        scalars = []
        triangles = []
        scatter_name = self.get_grid_name_from_uuid(geom_uuid)
        if scatter_name is not None:
            self.logger.info(f'Retrieving scatter set data for {scatter_name}...')
            scat = self._get_input_ugrid(scatter_name)
            if scat:
                points = self._get_input_ugrid_points(scatter_name, scat)
                triangles = self._get_input_ugrid_tris(scatter_name, scat)

                dset_name = self.get_dataset_name_from_uuid(dset_uuid)
                if dset_name is not None:
                    dset_reader = self.get_input_dataset(dset_name)
                    if dset_reader is not None:
                        scalars = dset_reader.values[0]
        return points, scalars, triangles

    def visit_node(self, poly_id, node):
        """Called to determine if arguments are valid.

        Args:
            poly_id (int): ID for polygon containing the node.
            node: The node.

        """
        node_id = node['id']
        if node_id in self._poly_nodes_visited:
            return

        geom = node['geometry']
        if self._polygon_node_merge[poly_id][node_id] == self.NODESPLIT:
            self._poly_corner_locs.append((geom.x, geom.y, geom.z))
        elif self._polygon_node_merge[poly_id][node_id] == self.NODEDEGEN:
            self._poly_corner_locs.append((geom.x, geom.y, geom.z))
            self._poly_corner_locs.append((geom.x, geom.y, geom.z))

        self._poly_nodes_visited.add(node_id)

    def are_points_equal(self, pt1, pt2):
        """Called to determine if arguments are valid.

        Args:
            pt1 (tuple): x,y,z location.
            pt2 (tuple): x,y,z location.

        Returns:
            (bool): True if equal.
        """
        dx = pt1[0] - pt2[0]
        dy = pt1[1] - pt2[1]
        dist = math.sqrt(dx * dx + dy * dy)
        if dist < 1.0e-9:
            return True
        return False

    def find_loc_in_list(self, pt, points):
        """Called to determine if arguments are valid.

        Args:
            pt (tuple): x,y,z location.
            points (list): The list of points as x,y,z tuples

        Returns:
            (int): First index in the list, -1 if not found.
        """
        idx = 0
        found = False
        for loc in points:
            if self.are_points_equal(pt, loc):
                found = True
                break
            idx += 1

        # return -1 if not found
        rval = -1
        if found is True:
            rval = idx
        return rval

    def setup_patching(self, poly, poly_input, nodes, arcs):
        """Called to determine if arguments are valid.

        Args:
            poly (NamedTuple): Poly for the input.
            poly_input (meshing.PolyInput): Input for meshing the polygon
            nodes (dict): Node DataFrame data as a dict, keyed by feature id
            arcs (dict): Arc DataFrame data as a dict, keyed by feature id

        Returns:
            (meshing.PolyInput): Input for meshing the polygon.
            (bool): Continue with meshing
        """
        self._poly_nodes_visited.clear()
        self._poly_corner_locs.clear()
        for arc_id in reversed(poly.polygon_arc_ids):
            # check both start and end nodes because arcs can go in different directions
            arc = arcs[arc_id]
            self.visit_node(poly.id, nodes[arc['start_node']])
            self.visit_node(poly.id, nodes[arc['end_node']])

        if len(self._poly_corner_locs) < 4:
            self.logger.warning('Error determining corners.')
            return poly_input, False

        if len(self._poly_corner_locs) > 4:
            corner_angles = []
            self.logger.warning(f'Polygon {poly.id} has too many nodes at corners for patching.  Attempting to keep '
                                'corners closest to 90 degrees.')
            # keep the "corners" that are closest to 90 degrees
            for loc in self._poly_corner_locs:
                cur = self.find_loc_in_list(loc, poly_input.outside_polygon)
                prev = cur - 1 if cur != 0 else len(poly_input.outside_polygon) - 1
                next = cur + 1 if cur != len(poly_input.outside_polygon) - 1 else 0
                angle = angle_between_edges_2d(poly_input.outside_polygon[prev], poly_input.outside_polygon[cur],
                                               poly_input.outside_polygon[next])
                diff = abs(math.pi - abs(angle))
                corner_angles.append((diff, loc, cur))

            # sort by angle difference
            corner_angles = sorted(corner_angles, key=lambda x: x[0], reverse=True)
            # keep the four biggest
            corner_angles = corner_angles[:4]
            # sort by index number so that these are in order around the polygon
            corner_angles = sorted(corner_angles, key=lambda x: x[2])
            # repopulate poly_corner_locs
            self._poly_corner_locs.clear()
            for i in range(4):
                self._poly_corner_locs.append(corner_angles[i][1])

        # set the first point to be the first corner
        shift_index = self.find_loc_in_list(self._poly_corner_locs[0], poly_input.outside_polygon)
        if shift_index != 0:
            poly_input.outside_polygon = np.roll(poly_input.outside_polygon, -shift_index, 0)

        # now we can assume that the first point is the first corner and go from there
        corners = [
            self.find_loc_in_list(self._poly_corner_locs[1], poly_input.outside_polygon),
            self.find_loc_in_list(self._poly_corner_locs[2], poly_input.outside_polygon),
            self.find_loc_in_list(self._poly_corner_locs[3], poly_input.outside_polygon),
        ]
        corners.sort()

        poly_input.polygon_corners = corners
        return poly_input, True

    def advance_index(self, cur_index, num_items):
        """Called to determine if arguments are valid.

        Args:
            cur_index (int): index to be advanced.
            num_items (int): number of items.

        Returns:
            (int): new index
        """
        if cur_index == num_items - 1:
            return 0
        else:
            return cur_index + 1

    def do_loc_lists_match(self, grid_locs, poly_locs):
        """Called to determine if arguments are valid.

        Args:
            grid_locs (list): grid_locations.
            poly_locs (list): poly locations.

        Returns:
            (bool): Locations match
        """
        # check that all grid locations are contained in the poly locations, and that all of the poly locations
        # are contained in the grid locations
        return self._loc_list_in_list(grid_locs, poly_locs) and self._loc_list_in_list(poly_locs, grid_locs)

    def _loc_list_in_list(self, list_1, list_2):
        """Called to determine if arguments are valid.

        Args:
            list_1 (list): locations to see if they are contained in the other list.
            list_2 (list): other list of locations.

        Returns:
            (bool): Locations match
        """
        rtree = index.Index()
        for i, p in enumerate(list_1):
            rtree.insert(i, (p[0], p[1], p[0], p[1]))
        bb = rtree.bounds
        xy_tol = xy_distance((bb[0], bb[1]), (bb[2], bb[3])) * 1.0e-8
        for loc in list_2:
            idx = list(rtree.nearest((loc[0], loc[1])))[0]
            poly_loc = list_1[idx]
            dist = xy_distance(loc, poly_loc)
            if dist > xy_tol:
                return False

        return True

    def do_poly_and_grid_match(self, grid, grid_name, poly_input):
        """Called to determine if arguments are valid.

        Args:
            grid (CoGrid): grid to merge
            grid_name (str): name of grid
            poly (meshing.PolyInput): PolyInput for current polygon.

        Returns:
            (bool): Boundaries match
        """
        ds_vals = [0] * grid.ugrid.cell_count
        poly_builder = GridCellToPolygonCoverageBuilder(co_grid=grid,
                                                        dataset_values=ds_vals,
                                                        wkt=None, coverage_name='temp',
                                                        logger=self.logger)
        out_polys = poly_builder.find_polygons()

        if len(out_polys[0]) != 1:
            self.logger.info(f'{self._geom_txt.capitalize()} invalid for this feature.')
            return False

        # are the points in the same locations?
        pts = grid.ugrid.locations
        grid_boundary = [(pts[idx][0], pts[idx][1], pts[idx][2]) for idx in out_polys[0][0][0]]

        if self.do_loc_lists_match(grid_boundary, poly_input.outside_polygon) is False:
            self.logger.warning(f'The boundary for "{grid_name}" does not match polygon {poly_input.polygon_id}. '
                                f'Polygon {poly_input.polygon_id} will not be used.')
            return False

        return True

    def poly_input_from_id(self, poly, poly_pts, nodes, arcs):
        """Called to determine if arguments are valid.

        Args:
            poly (NamedTuple): Poly for the input.
            poly_pts (list(list)): The outer and inner rings that make up the polygon.
            nodes (dict): Node DataFrame data as a dict keyed by feature id. Used for patching.
            arcs (dict): Arc DataFrame data as a dict keyed by feature id. Used for patching.

        Returns:
            (bool) (meshing.PolyInput)- Whether or not to do meshing for the poly, poly input

        """
        outer_poly = poly_pts[0][:-1]  # remove the last point because it is the same as the first
        # outer_pts_2d = [(p[0], p[1], 0.0) for p in outer_poly]
        # sh_poly = Polygon(pts_2d)
        # bb = sh_poly.bounds
        # dx = bb[0] - bb[2]
        # dy = bb[1] - bb[3]
        # tol = math.sqrt(dx ** 2 + dy ** 2) * 1e-9
        # sh_poly = self._buffer_polygon(sh_poly, tol)
        # outer_pts_2d = [(p[0], p[1]) for p in sh_poly.exterior.coords]
        # # outer_path = path.Path(pts_2d)

        inner_poly_list = []
        if len(poly_pts) > 1:
            inner_poly_extra_pt = poly_pts[1:]
            for inner_poly in inner_poly_extra_pt:
                inner_poly_list.append(inner_poly[:-1])  # remove the last point - it's a duplicate
                # pts_2d = [(p[0], p[1]) for p in inner_poly_list[-1]]
                # sh_poly = Polygon(pts_2d)
                # sh_poly = self._buffer_polygon(sh_poly, tol)
                # inner_pts_2d = [(p[0], p[1]) for p in sh_poly.exterior.coords]
                # inner_polys.append(inner_pts_2d)

        # add interior arcs
        remove_arcs = set()
        for arc_id, arc in self._disjoint_arcs.items():
            arc_pts = list(arc.geometry.coords)
            num_pts = len(arc_pts)
            arc_locs = [(x, y, 0.0) for (x, y, z) in arc_pts]
            num_on_poly = 0
            arc_in_poly = True
            for arc_loc in arc_locs:
                result = point_in_polygon_2d(outer_poly, arc_loc)
                if result == -1:
                    arc_in_poly = False
                if result == 0:
                    num_on_poly += 1
                    arc_in_poly = num_on_poly < num_pts
                if not arc_in_poly:
                    break

            if arc_in_poly:
                for inner in inner_poly_list:
                    num_on_poly = 0
                    if not arc_in_poly:
                        break
                    for arc_loc in arc_locs:
                        result = point_in_polygon_2d(inner, arc_loc)
                        if result == 1:
                            arc_in_poly = False
                        if result == 0:
                            num_on_poly += 1
                            arc_in_poly = num_on_poly < num_pts
                        if not arc_in_poly:
                            break

                if arc_in_poly:
                    remove_arcs.add(arc_id)
                    arc_locs = [(x, y, z) for x, y, z in arc_pts]
                    if len(arc_locs) >= 2:
                        rev_locs = arc_locs[1:-1]
                        rev_locs.reverse()
                        inner_poly_list.append(arc_locs + rev_locs)
        for arc_id in remove_arcs:
            self._disjoint_arcs.pop(arc_id)

        poly_input = meshing.PolyInput(outside_polygon=outer_poly, remove_internal_four_triangle_points=True,
                                       inside_polygons=inner_poly_list, bias=0.3, fix_point_connections=True,
                                       polygon_id=poly.id)

        if self._mesh_cov is None:
            return True, poly_input

        idx = self._poly_ids.index(poly.id)
        meshing_type = self._mesh_type[idx]

        if meshing_type == self.MESH_UGRID_BRIDGE:
            uuid = self._ugrid_bridge_uuid[idx]
            grid_name = self.get_grid_name_from_uuid(uuid)
            if grid_name is not None:
                grid = self.get_input_grid(grid_name)
                if grid is not None and self.do_poly_and_grid_match(grid, grid_name, poly_input):
                    self._ugrids_to_merge.append(grid)
            return False, poly_input

        bias = self._bias[idx]
        if bias >= 0 and bias <= 1.0:
            poly_input.bias = bias

        if meshing_type == self.MESH_CPAVE:
            override_size = self._cell_size[idx]
            if override_size >= 0.0:
                poly_input.constant_size_bias = bias
                poly_input.constant_size_function = override_size
                if gu.valid_wkt(self.default_wkt) and gu.is_geographic(self.default_wkt):
                    np_outer_poly = np.array(outer_poly)
                    length = np_outer_poly.shape[0]
                    sum_y = np.sum(np_outer_poly[:, 1])
                    lat = sum_y / length
                    # There are about 111111 meters per degree of latitude
                    lat_cos = math.cos(math.radians(lat))
                    poly_input.constant_size_function = poly_input.constant_size_function / 111111.0 * lat_cos
        if meshing_type == self.MESH_UGRID_PATCH and self._force_ugrid is False:
            self.fail(f'A UGrid patch cannot be used when creating a 2d mesh. '
                      f'Change the type for polygon: {poly.id}. Aborting.')

        if meshing_type == self.MESH_PATCH or meshing_type == self.MESH_UGRID_PATCH:
            poly_input, do_mesh = self.setup_patching(poly, poly_input, nodes, arcs)
            if do_mesh is False:
                return False, poly_input
            if meshing_type == self.MESH_UGRID_PATCH:
                self._ugrid_patch_polys.append(poly_input)

        if meshing_type == self.MESH_SPAVE:
            poly_input.size_function = self._get_interpolator(idx, is_spave=True)

        # bathy
        poly_input = self.set_bathy_input(poly_input, idx)

        # refine points in the polygon
        have_pts = self._all_disjoint_pts is not None and len(self._all_disjoint_pts.index) > 0
        if meshing_type != self.MESH_PATCH and have_pts:
            poly_locs = poly_input.outside_polygon[:, :-1]  # Drop the Z-coordinate
            poly_locs = np.append(poly_locs, [poly_locs[0]], axis=0)
            in_poly = run_parallel_points_in_polygon(self._all_disjoint_locs, poly_locs)
            for i, is_in in enumerate(in_poly):
                if is_in:
                    the_pt = self._all_disjoint_pts.iloc[i]
                    idx = self._point_ids.index(the_pt.id)
                    if bool(self._refine_on[idx]) is True:
                        self._refine_pts.append(RefinePoint(point=(the_pt.geometry.x, the_pt.geometry.y,
                                                                   the_pt.geometry.z),
                                                            size=self._element_size[idx], create_mesh_point=True))
                    else:
                        self.logger.warning(
                            f'Feature point {the_pt.id} was skipped: Creating a mesh node without it being a refine '
                            'point is not supported.'
                        )

        return True, poly_input

    def interp_raster_elevs(self, pts):
        """Interpolate raster elevations.

        Args:
            pts (list): list of points on the mesh.

        """
        self._points_list = np.asarray([(p[0], p[1]) for p in pts])
        assigned = set()
        for raster in self._raster_to_poly_input:
            raster_name = self.get_raster_name_from_uuid(raster)
            if raster_name is None:
                self.fail(f'Error retrieving raster: {raster}')

            raster_file = self.get_input_raster_file(raster_name)
            elevations, no_data = ru.interpolate_raster_to_points(raster_file, pts,
                                                                  self._mesh_cov.crs.
                                                                  to_wkt(version=WktVersion.WKT1_GDAL),
                                                                  self.vertical_units, interpolate_locations=True)

            # get array for whether points are in the polygons
            in_polys = list(self._get_points_in_polys(self._raster_to_poly_input[raster]))

            for i in range(len(pts)):
                # find if it is in a polygon that this raster is assigned to
                if in_polys[i]:
                    if elevations[i] != no_data:
                        assigned.add(i)  # Keep track of the points we have interpolated elevations to.
                    pts[i][2] = elevations[i]

        self._report_uninterpolated_elevs(assigned)

    def _report_uninterpolated_elevs(self, assigned):
        """Tell the user about any points who did not get an interpolated raster elevation.

        Args:
            assigned (set): 0-based indices of the points that were assigned elevations
        """
        if len(assigned) < len(self._points_list):
            all_pts = set(range(len(self._points_list)))
            # Switch to a 1-based numpy array for the user's viewing pleasure.
            unassigned = np.array(list(all_pts - assigned)) + 1
            self.logger.warning('The following points were outside the bounds of the elevation raster and were not '
                                f'assigned Z-values:\n{unassigned}')

    def _get_points_in_polys(self, input_polys):
        """Interpolate raster elevations.

        Args:
            input_polys (list): list of polygon points for this raster

        Return:
            in_poly (list): list of bools if point is in a poly

        """
        in_poly = None
        for input_poly in input_polys:
            poly_locs = input_poly.outside_polygon[:, :2]
            poly_locs = np.append(poly_locs, [poly_locs[0]], axis=0)
            out = run_parallel_points_in_polygon(self._points_list, poly_locs)

            for inside_poly in input_poly.inside_polygons:
                inner_poly_locs = inside_poly[:, :2]
                inner_poly_locs = np.append(inner_poly_locs, [inner_poly_locs[0]], axis=0)
                inner_out = run_parallel_points_in_polygon(self._points_list, inner_poly_locs)

                # 2's are on the edge - in this case we want them to count as NOT in the inner poly, so change
                # the 2's to 0's
                inner_out[inner_out == 2] = 0

                # if we are in both the outer and an inner polygon, we want it to be false
                out = np.logical_xor(out, inner_out)

            if in_poly is None:
                in_poly = out
            else:
                in_poly = np.logical_or(in_poly, out)

        return in_poly

    def edge_index(self, edges, edge):
        """Called for ugrid patch.

        Args:
            edges (list): list of tuples of cell edges
            merge_edge (tuple): edge

        Returns:
            (int): index of edge

        """
        index = -1
        found = False
        i = 0
        for cur_edge in edges:
            if cur_edge == edge or (cur_edge[0] == edge[1] and cur_edge[1] == edge[0]):
                found = True
                break
            i += 1

        if found:
            index = i

        return index

    def get_tri_info_for_ugrid_patch(self, ug, index):
        """Called for ugrid patch.

        Args:
            ug (UGrid): the grid to modify.
            index (int): the cell index
        """
        pts = ug.get_cell_points(index)
        locs = ug.get_points_locations(pts)
        angles = [abs(angle_between_edges_2d(locs[0], locs[1], locs[2])),
                  abs(angle_between_edges_2d(locs[1], locs[2], locs[0])),
                  abs(angle_between_edges_2d(locs[2], locs[0], locs[1]))]
        angles.sort()
        min_angle = angles[0]

        if min_angle not in self._small_angle_to_id.keys():
            self._small_angle_to_id[min_angle] = []
        self._small_angle_to_id[min_angle].append(index)

        longest_edge = xy_distance(locs[0], locs[1])
        longest_pts = (pts[0], pts[1])
        length = xy_distance(locs[1], locs[2])
        if length > longest_edge:
            longest_edge = length
            longest_pts = (pts[1], pts[2])
        length = xy_distance(locs[2], locs[0])
        if length > longest_edge:
            longest_pts = (pts[2], pts[0])

        self._cell_id_to_long_edge[index] = longest_pts

    def _get_angle_info_for_tri(self, ug, index):
        """Called for ugrid patch.

        Args:
            ug (UGrid): the grid to modify.
            index (int): the cell index
        """
        point_id_to_angle = {}
        pts = ug.get_cell_points(index)
        locs = ug.get_points_locations(pts)

        # dictionary of the triangle point matched with (angle at pt, definition of opposite edge)
        angle_1 = abs(angle_between_edges_2d(locs[2], locs[1], locs[0])) * (180 / math.pi)
        point_id_to_angle[pts[1]] = (angle_1, (pts[0], pts[2]))
        angle_2 = abs(angle_between_edges_2d(locs[0], locs[2], locs[1])) * (180 / math.pi)
        point_id_to_angle[pts[2]] = (angle_2, (pts[1], pts[0]))
        angle_3 = abs(angle_between_edges_2d(locs[1], locs[0], locs[2])) * (180 / math.pi)
        point_id_to_angle[pts[0]] = (angle_3, (pts[2], pts[1]))

        return point_id_to_angle

    def merge_triangle_for_ugrid_patch(self, ug, index):
        """Called for ugrid patch.

        Args:
            ug (UGrid): the grid to modify.
            index (int): the cell index
        """
        if index not in self._id_to_new_cellstream:
            merge_edge = self._cell_id_to_long_edge[index]
            tri_edges = ug.get_cell_edges(index)
            next_index = tri_merge_index = self.edge_index(tri_edges, merge_edge)

            adjs = ug.get_edge_adjacent_cells(merge_edge)
            # the longest edge is on the border, so there is no adjacent cell
            while len(adjs) != 2:
                # get the next longest edge
                next_idx = self.advance_index(next_index, 3)
                length = xy_distance(ug.locations[tri_edges[next_idx][0]], ug.locations[tri_edges[next_idx][1]])
                merge_edge = tri_edges[next_idx]
                tri_merge_index = next_idx

                next_idx = self.advance_index(next_idx, 3)
                if xy_distance(ug.locations[tri_edges[next_idx][0]], ug.locations[tri_edges[next_idx][1]]) > length:
                    merge_edge = tri_edges[next_idx]
                    tri_merge_index = next_idx

                adjs = ug.get_edge_adjacent_cells(merge_edge)

            for id in adjs:
                if id != index:
                    adj_id = id
                    break

            adj_edges = []
            new_cellstream_id = len(self._id_to_new_cellstream)

            # get the edges...has this already been merged?
            if adj_id in self._id_to_new_cellstream.keys():
                new_cellstream_id = self._id_to_new_cellstream[adj_id]
                cellstream = self._new_cellstreams[new_cellstream_id]
                cellstream[0] = 7  # set to polygon
                cellstream[1] = cellstream[1] + 2  # add two more sides for the triangle we are merging with

                for i in range(2, len(cellstream) - 1):
                    adj_edges.append((cellstream[i], cellstream[i + 1]))
                adj_edges.append((cellstream[len(cellstream) - 1], cellstream[2]))

                num_adj_edges = len(adj_edges)
            else:
                adj_edges = ug.get_cell_edges(adj_id)
                num_adj_edges = len(adj_edges)

            # merge it
            adj_merge_index = self.edge_index(adj_edges, merge_edge)

            new_cellstream = []
            new_cellstream.extend([7, len(adj_edges) + 1, merge_edge[1]])

            tri_index = self.advance_index(tri_merge_index, 3)
            new_cellstream.extend([tri_edges[tri_index][1], merge_edge[0]])

            adj_index = self.advance_index(adj_merge_index, num_adj_edges)
            for _ in range(num_adj_edges - 2):
                new_cellstream.append(adj_edges[adj_index][1])
                adj_index = self.advance_index(adj_index, num_adj_edges)

            self._new_cellstreams[new_cellstream_id] = new_cellstream

            # set the cellstream ids
            self._id_to_new_cellstream[index] = new_cellstream_id
            self._id_to_new_cellstream[adj_id] = new_cellstream_id

    def eliminate_triangles_for_ugrid_patch(self, ug):
        """Called for ugrid patch.

        Args:
            ug (UGrid): the grid to modify.

        Returns:
            (UGrid): the modified grid.

        """
        for i in range(ug.cell_count):
            if ug.get_cell_edge_count(i) == 3:
                for poly_input in self._ugrid_patch_polys:
                    pt = ug.get_cell_centroid(i)
                    if pt:
                        poly_locs = np.asarray([(p[0], p[1]) for p in poly_input.outside_polygon])
                        poly_locs = np.append(poly_locs, [poly_locs[0]], axis=0)
                        pt_array = np.array([pt[1][0], pt[1][1]])
                        if bool(_is_inside_sm(poly_locs, pt_array)) is True:
                            self.get_tri_info_for_ugrid_patch(ug, i)
                            break

        angles = sorted(self._small_angle_to_id.keys())
        for angle in angles:
            for tri in self._small_angle_to_id[angle]:
                self.merge_triangle_for_ugrid_patch(ug, tri)

        old_cell_count = ug.cell_count
        old_cellstream = np.asarray(ug.cellstream)
        new_cellstream = []

        cur_old_cell_id = 0
        stream_idx = 0

        while cur_old_cell_id < old_cell_count:
            num_cell_pts = old_cellstream[stream_idx + 1]
            next_cell_stream_idx = stream_idx + num_cell_pts + 2

            # only copy if the cell was not modified
            if cur_old_cell_id not in self._id_to_new_cellstream.keys():
                new_cellstream.extend(old_cellstream[stream_idx:next_cell_stream_idx])

            cur_old_cell_id += 1
            stream_idx = next_cell_stream_idx

        for stream in self._new_cellstreams:
            new_cellstream.extend(self._new_cellstreams[stream])

        # create the new ugrid
        ug = XmUGrid(ug.locations, new_cellstream)

        return ug

    def _get_cell_min_angle_info(self, ug, tri):
        """Called for ugrid patch.

        Args:
            ug (UGrid): the grid to modify.
            tri (int): triangle cell id

        Returns:
            (dict): dictionary of angle info

        """
        if tri not in self._cell_id_to_min_angle_info.keys():
            return None
        info = self._cell_id_to_min_angle_info[tri]
        if info is None:
            self._cell_id_to_min_angle_info[tri] = info = self._get_angle_info_for_tri(ug, tri)
        return info

    def _merge_tris_by_min_angle(self, ug):
        """Called for ugrid patch.

        Args:
            ug (UGrid): the grid to modify.

        Returns:
            (UGrid): the modified grid.

        """
        self._cell_id_to_min_angle_info = {i: None for i in range(ug.cell_count) if ug.get_cell_edge_count(i) == 3}
        id_to_new_cellstream = {}

        tri_ids = list(self._cell_id_to_min_angle_info.keys())
        for tri in tri_ids:
            info = self._get_cell_min_angle_info(ug, tri)
            if info is None:
                continue
            for pt in info.keys():
                if info[pt][0] > self._merge_angle:
                    # this is an edge that we could potentially merge - get the other side
                    merge_edge = info[pt][1]
                    adj_cells = ug.get_edge_adjacent_cells(merge_edge)
                    if len(adj_cells) == 1:
                        # must be on the boundary
                        continue
                    opposite_tri = adj_cells[0] if adj_cells[1] == tri else adj_cells[1]
                    opposite_info = self._get_cell_min_angle_info(ug, opposite_tri)
                    if opposite_info is None:
                        # this cell is either not a triangle or has already been processed
                        continue
                    opp_pts = opposite_info.keys()
                    the_pt = -1
                    for opp_pt in opp_pts:
                        if opp_pt != merge_edge[0] and opp_pt != merge_edge[1]:
                            the_pt = opp_pt
                            break

                    opposite_angle = opposite_info[the_pt][0]
                    if opposite_angle < self._merge_angle:
                        # this will leave an angle smaller than the minimum
                        continue

                    # now check the see if the next angles added together
                    tri_angle = info[merge_edge[0]][0]
                    opp_tri_angle = opposite_info[merge_edge[0]][0]
                    if (tri_angle + opp_tri_angle) < self._merge_angle:
                        continue

                    # check on the other angles
                    tri_angle = info[merge_edge[1]][0]
                    opp_tri_angle = opposite_info[merge_edge[1]][0]
                    if (tri_angle + opp_tri_angle) < self._merge_angle:
                        continue

                    # if we got here, we will merge!
                    id_to_new_cellstream[tri] = [7, 4, pt, merge_edge[1], the_pt, merge_edge[0]]
                    id_to_new_cellstream[opposite_tri] = []

                    # remove them from the dictionary so that we don't try to merge with other tris
                    del self._cell_id_to_min_angle_info[tri]
                    del self._cell_id_to_min_angle_info[opposite_tri]
                    break

        if len(id_to_new_cellstream.keys()) == 0:
            return ug

        old_cell_count = ug.cell_count
        old_cellstream = np.asarray(ug.cellstream)
        new_cellstream = []

        cur_old_cell_id = 0
        stream_idx = 0

        while cur_old_cell_id < old_cell_count:
            num_cell_pts = old_cellstream[stream_idx + 1]
            next_cell_stream_idx = stream_idx + num_cell_pts + 2

            # only copy if the cell was not modified
            if cur_old_cell_id not in id_to_new_cellstream.keys():
                new_cellstream.extend(old_cellstream[stream_idx:next_cell_stream_idx])
            else:
                new_cellstream.extend(id_to_new_cellstream[cur_old_cell_id])

            cur_old_cell_id += 1
            stream_idx = next_cell_stream_idx

        # create the new ugrid
        ug = XmUGrid(ug.locations, new_cellstream)

        return ug

    def _calc_ugrid_xy_tol(self, ugrid):
        """Calculate an xy tolerance for the ugrid.

        Args:
            ugrid (UGrid): unstructured grid
        """
        extents = ugrid.extents
        dx = extents[0][0] - extents[1][0]
        dy = extents[0][1] - extents[1][1]
        length = math.sqrt(dx ** 2 + dy ** 2)
        return length * 1.0e-9

    def _clear_cache(self):
        """Release cached memory."""
        self._input_scatters = {}
        self._input_scatter_points = {}
        self._input_scatter_cells = {}

    # def _buffer_polygon(self, sh_poly, buffer_dist):
    #     """Workaround for buffer polygon bug. Described here: https://github.com/shapely/shapely/issues/1044.
    #
    #     Args:
    #         sh_poly (Polygon): polygon to buffer
    #         buffer_dist (float): distance to buffer
    #
    #     Returns:
    #         (Polygon): buffered poly
    #     """
    #     buffered_poly = sh_poly.buffer(distance=buffer_dist)
    #     if buffered_poly.geom_type == 'MultiPolygon':
    #         biggest_buffered_poly = sorted(buffered_poly.geoms, key=lambda g: g.area)[-1]
    #         buffered_poly = biggest_buffered_poly
    #     return buffered_poly

    def _check_poly_pts(self, poly, poly_pts):
        """Ensure valid number of polygon points.

        Args:
            poly (DataFrame): The polygon
            poly_pts (list): List of polygon points
        """
        # check for invalid polygons
        for pts in poly_pts:
            if len(pts) < 4:
                self.fail(f'Invalid polygon detected in coverage, id: {poly.id}. Clean the '
                          'coverage and rebuild polygons. Aborting.')

    def _report_and_pause(self, msg):
        """Log a message from a C++ library and sleep the thread to allow the GUI to update.

        Args:
            msg (str): The log message.
        """
        now = time.time()
        if now - self.start_time > self.log_interval:
            self.start_time = now
            self.logger.info(msg)

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        self.logger.info('Retrieving input meshing coverage...')
        self._poly_cov = self.get_input_coverage(arguments[ARG_INPUT_COVERAGE].value)
        self._merge_triangles = arguments[ARG_INPUT_MERGE_TRIS].value

        # Give output mesh the input coverage's name if not specified.
        if arguments[ARG_INPUT_MESH_NAME].text_value == '':
            arguments[ARG_INPUT_MESH_NAME].value = arguments[ARG_INPUT_COVERAGE].value.split('/')[-1]

        # find out if this is a mesh generator coverage
        coverage = self._poly_cov
        atts = coverage.attrs['attributes'] if coverage is not None else None
        if atts is not None and 'CoverageType' in atts and atts['CoverageType'] == 'MESH_GENERATION':
            self._mesh_cov = coverage
            self.get_mesh_cov_props()

        if not self._poly_cov.empty:
            meshing_inputs = []
            # make a list of all arcs that are not connected to polygons
            poly_arcs = set()
            self.logger.info('Checking input polygons...')
            polygons = self._poly_cov[self._poly_cov['geometry_types'] == 'Polygon']
            for poly in polygons.itertuples():
                if not poly.geometry.is_valid:
                    self.fail(f'Invalid polygon found with id {poly.id}\n{explain_validity(poly.geometry)}')
                for arc_id in poly.polygon_arc_ids:
                    poly_arcs.add(arc_id)
                for arc_id_list in poly.interior_arc_ids:
                    for arc_id in arc_id_list:
                        poly_arcs.add(arc_id)
            arcs = self._poly_cov[self._poly_cov['geometry_types'] == 'Arc']
            for arc in arcs.itertuples():
                if arc.id not in poly_arcs:
                    self._disjoint_arcs[arc.id] = arc

            # Convert arc and node data to dicts keyed by feature id. Don't want to iterate through the DataFrame.
            self.logger.info('Extracting features from input coverage...')
            arcs.set_index('id', inplace=True, drop=False)
            arcs = arcs[['start_node', 'end_node']].to_dict('index')
            nodes = self._poly_cov[self._poly_cov['geometry_types'] == 'Node']
            nodes.set_index('id', inplace=True, drop=False)
            nodes = nodes.to_dict('index')

            # Need to do this because we don't have any dataset arguments to the tool but individual polys may be
            # using a scatter's elevation as a bathy source.
            self._data_handler.get_available_datasets()
            self.logger.info('Gathering meshing properties from polygons...')
            for poly in polygons.itertuples():
                try:
                    idx = self._poly_ids.index(poly.id)
                    if self._mesh_type[idx] == self.MESH_NONE:
                        continue  # This is a disabled polygon in a meshing coverage, don't do any more work.
                except ValueError:
                    pass

                poly_pts = get_polygon_point_lists(poly)
                self._check_poly_pts(poly, poly_pts)

                do_meshing, poly_input = self.poly_input_from_id(poly, poly_pts, nodes, arcs)

                if do_meshing:
                    meshing_inputs.append(poly_input)
                    self._poly_count += 1
            self._clear_cache()

            if self._poly_count == 0 and len(self._ugrids_to_merge) == 0:
                self.logger.info('No valid polygons in coverage to mesh.')
                return

            b = UGridBuilder()
            b.set_is_2d()
            ug = None
            if self._poly_count > 0:
                self.logger.info('Generating mesh...')
                ug = xms.mesher.generate_mesh(polygon_inputs=meshing_inputs, refine_points=self._refine_pts,
                                              progress_callback=self._report_and_pause)

                if len(self._raster_to_poly_input) > 0:
                    pts = ug.locations
                    self.interp_raster_elevs(pts)
                    ug.locations = pts

            if len(self._ugrids_to_merge):
                if ug:
                    b.set_ugrid(ug)

                for grid in self._ugrids_to_merge:
                    if ug is None:
                        ug = grid.ugrid
                    else:
                        b.set_ugrid(ug)
                        new_mesh = b.build_grid()
                        ug_tol = self._calc_ugrid_xy_tol(ug)
                        grid_tol = self._calc_ugrid_xy_tol(grid.ugrid)
                        pt_tol = ug_tol if ug_tol > grid_tol else grid_tol
                        merger = UGrid2dMerger(grid_1=grid, grid_2=new_mesh, pt_tol=pt_tol, logger=self.logger,
                                               stitch_grids=True)
                        ug = merger.merge_grids()

            if len(self._ugrid_patch_polys) > 0:
                ug = self.eliminate_triangles_for_ugrid_patch(ug)
                b.set_unconstrained()

            if self._merge_triangles:
                self._merge_angle = arguments[ARG_INPUT_MERGE_TRIS_ANGLE].value
                ug = self._merge_tris_by_min_angle(ug)

            b.set_ugrid(ug)
            output = b.build_grid()
            wkt = None
            if self._poly_cov.crs is not None:
                wkt = self._poly_cov.crs.to_wkt(WktVersion.WKT1_GDAL)
            if not wkt and self.default_wkt:
                wkt = xms.gdal.utilities.gdal_utils.add_vertical_to_wkt(self.default_wkt, self.vertical_datum,
                                                                        self.vertical_units)
            self.set_output_grid(output, arguments[ARG_INPUT_MESH_NAME], wkt, force_ugrid=self._force_ugrid)
