"""Base class for dataset Tools."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from collections.abc import Iterable
from typing import Tuple

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.datasets.dataset_writer import DatasetWriter

# 4. Local modules


def set_builder_activity_flags(activity_array, builders):
    """Set the output dataset builder flags indicating whether or not to use the activity array as null values.

    I wanted this method for centralized test coverage purposes.

    Args:
        activity_array (Optional[Sequence[int]]): The output activity array, None if not used
        builders (Union[DatasetWriter, Sequence[DatasetWriter]): The output dataset builders
    """
    if not isinstance(builders, Iterable):  # Allow passing a single DatasetReader.
        builders = [builders]

    use_activity = activity_array is not None
    for builder in builders:
        if builder is not None:
            builder.use_activity_as_null = use_activity


def get_min_max(values: np.ndarray) -> Tuple[float, float]:
    """Get the minimum and maximum values of a numpy array.

    Args:
        values (np.ndarray): The array.

    Returns:
        (Tuple[float, float]): The minimum and maximum value.
    """
    minimum = np.fmin.reduce(values)
    if np.isnan(minimum):
        minimum = np.inf
    maximum = np.fmax.reduce(values)
    if np.isnan(maximum):
        maximum = -np.inf
    return minimum, maximum


def smooth_get_ugrid(tool, dset_arg):
    """Gets the ugrid for the Smooth Datasets tool.

    Args:
        tool (xms.tool_core.Tool): The tool.
        dset_arg (DatasetArgument): The dataset argument.

    Returns:
        (UGrid): The ugrid associated with the input dataset.
    """
    dset_grid = tool.get_input_dataset_grid(dset_arg.text_value)
    if dset_grid is None:
        tool.fail("Unable to read dataset input's geometry.")
    dset_ugrid = dset_grid.ugrid
    if dset_ugrid.cell_count < 1:
        msg = 'No cells (triangles) defined for the grid associated with the input dataset. ' \
              'This tool requires cells to smooth the dataset values.'
        tool.fail(msg)
    # loop through all ugrid points and give a message about points that are not connected
    # to any cells
    disjoint_ids = []
    for idx in range(dset_ugrid.point_count):
        cells = dset_ugrid.get_point_adjacent_cells(idx)
        if cells is None or len(cells) < 1:
            disjoint_ids.append(idx + 1)
    if disjoint_ids:
        msg = f'Some points in the UGrid are not connected to any cells (triangles). The dataset value at these ' \
              f'points will not be smoothed.\nPoint ids: {disjoint_ids}'
        tool.logger.warning(msg)
    return dset_ugrid


def copy_datasets_to_new_geometry(inputs):
    """Copies datasets to a new geometry.

    Args:
        inputs (dict): dict of inputs (tool, old_geom_name, new_geom_uuid, datasets)
    """
    tool = inputs['tool']
    tool.logger.info('Creating datasets...')
    ug_name = inputs['old_geom_name']
    new_geom_uuid = inputs['new_geom_uuid']
    datasets = inputs['datasets']
    elevs = np.asarray([p[2] for p in inputs['ug_points']], dtype='float32')
    calculator = inputs['calculator']
    for dset_name in inputs['available_datasets']:
        # check to see if it is the elevation dataset before copying
        reader = tool.get_input_dataset(dset_name)
        if reader.num_components == 1:
            if reader.activity is not None and reader.location == 'points':
                reader.activity_calculator = calculator
            values = reader.timestep_with_activity(0)[0]
            if np.array_equal(elevs, values):
                continue

        tool.logger.info(f'Copying "{dset_name}" to output ugrid...')
        dset_name = dset_name.replace(f'{ug_name}/', '')
        writer = DatasetWriter(name=dset_name, geom_uuid=new_geom_uuid)
        writer.duplicate_from_reader(reader)
        datasets.append(writer)
