"""CoverageArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Dict, Optional, Tuple, Union

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument, contains_invalid_characters, IoDirection
from .data_handler import DataHandler

FiltersList = Optional[Union[Dict[str, str], Tuple[str, str]]]
# Filters are keyword-value pairs
ALLOW_ONLY_COVERAGE_TYPE = 'COVERAGE_TYPE'  # XML-defined coverage type of DMI coverage or generic coverage type name
ALLOW_ONLY_MODEL_NAME = 'MODEL_NAME'  # XML-defined model name of DMI coverage


class CoverageArgument(Argument):
    """Coverage tool argument."""

    def __init__(
            self,
            data_handler: DataHandler,
            name: str,
            description: str = '',
            io_direction: Optional[IoDirection] = IoDirection.INPUT,
            optional: Optional[bool] = False,
            value: Optional[str] = None,
            hide: Optional[bool] = False,
            filters: FiltersList = None
    ):
        """Construct a coverage argument.

        Args:
            data_handler (DataHandler): Data handler to give list of potential coverages.
            name (Optional[str]): Python friendly argument name.
            description (Optional[str]): User friendly description of the argument.
            io_direction (Optional[IoDirection]): IO Direction of the argument (input or output).
            optional (Optional[bool]): Is the argument optional?
            value (Optional[str]): Default value.
            hide (Optional[bool]): Should the argument be hidden (True) or visible?
            filters (FiltersList): List of filters for allowed coverage types.
        """
        super().__init__(name, description, io_direction, optional, value, hide)
        self.data_handler = data_handler
        if isinstance(filters, tuple):
            filters = {filters[0]: filters[1]}
        elif filters is None:
            filters = {}
        self.filters = filters

    def _get_type(self) -> str:
        """Get a string representing the argument type.

        Returns:
            (str): The argument type.
        """
        return 'coverage'

    def _set_value(self, value: str) -> None:
        """Set the argument value.

        Args:
            value (str): The new argument value.
        """
        self._value = None if value is None else str(value)

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        interface_info = {
            'type': 'StringSelector' if self.io_direction == IoDirection.INPUT else 'String'
        }
        interface_info = interface_info | super().get_interface_info()
        if self.io_direction == IoDirection.INPUT:
            coverage_list = self.data_handler.get_available_coverages(self.filters).copy()
            coverage_list.insert(0, Argument.NONE_SELECTED)
            if self.value not in coverage_list:
                interface_info['value'] = Argument.NONE_SELECTED
            interface_info['choices'] = coverage_list
        return interface_info

    def to_dict(self) -> dict:
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'CoverageArgument'}
        super_values = super(CoverageArgument, self).to_dict()
        values.update(super_values)
        values['filters'] = self.filters
        self._add_key_value('uuid', self.data_handler.get_uuid_from_coverage_name(self.text_value), values)
        return values

    def adjust_value_from_results(self) -> None:
        """Adjust the value obtained when running from previous results."""
        if self.io_direction == IoDirection.INPUT and \
                self.value not in self.data_handler.get_available_coverages(self.filters):
            self.value = None

    def validate(self) -> str:
        """Validate the argument.

        Returns:
            (str): An error string if invalid or None.
        """
        result = super().validate()
        if result is not None:
            return result

        coverage_name = self.value
        if self.data_handler.check_duplicate_output:
            if self.io_direction == IoDirection.OUTPUT:
                # Check if coverage already exists in the list of possible input coverages.
                # If it does exist, return an error message.
                if self.data_handler.coverage_exists(coverage_name):
                    return f'Coverage {coverage_name} already exists.  Please enter another coverage name.'
        return contains_invalid_characters(self)
