"""FloatArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument, IoDirection


class FloatArgument(Argument):
    """Floating point tool argument."""
    def __init__(
        self,
        name: str,
        description: str = '',
        io_direction: Optional[IoDirection] = IoDirection.INPUT,
        optional: Optional[bool] = False,
        value: Optional[float] = None,
        hide: Optional[bool] = False,
        min_value: Optional[float] = None,
        max_value: Optional[float] = None
    ):
        """Construct a dataset argument.

        Args:
            name (Optional[str]): Python friendly argument name.
            description (Optional[str]): User friendly description of the argument.
            io_direction (Optional[IoDirection]): IO Direction of the argument (input or output).
            optional (Optional[bool]): Is the argument optional?
            value (Optional[float]): Default value.
            hide (Optional[bool]): Should the argument be hidden (True) or visible?
            min_value (Optional[float]): The minimum value of the argument.
            max_value (Optional[float]): The maximum value of the argument.
        """
        super().__init__(name, description, io_direction, optional, value, hide)
        # need to make sure values are floats because JSON write/read can make whole numbers ints
        if self.value is not None:
            self.value = float(self.value)
        self._min_value = min_value
        if self._min_value is not None:
            self._min_value = float(self._min_value)
        self._max_value = max_value
        if self._max_value is not None:
            self._max_value = float(self._max_value)
        if self.io_direction == IoDirection.OUTPUT:
            self.optional = True  # output doesn't need to be specified

    def _get_type(self) -> str:
        """Get a string representing the argument type (float).

        Returns:
            (str): The argument type.
        """
        return 'float'

    def _set_value(self, value: Optional[float]) -> None:
        """Set the argument value.

        Args:
            value (Optional[float]): The new argument value.
        """
        self._value = None if value is None else float(value)

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        if self.io_direction == IoDirection.OUTPUT:
            return None
        interface_info = {'type': 'Number'}

        interface_info = interface_info | super().get_interface_info()
        if interface_info['value'] is None:
            interface_info['value'] = 0.0
        return interface_info

    def _is_required_satisfied(self) -> bool | None:
        """Is the "required" value satisfied?

        For a floating point argument, the GUI interface always has a value.

        Returns:
              True/False if the required value is satisfied. None if not required.
        """
        return None

    def validate(self) -> Optional[str]:
        """Validate the argument.

        Returns:
            (str): An error string if invalid or None.
        """
        if self.value is not None:
            # some sample_tools won't work unless a float argument value really is a float
            value = float(self.value)
            self.value = value
            if self._min_value is not None and value < self._min_value:
                return f'Value must be greater than or equal to {self._min_value}.'
            if self._max_value is not None and value > self._max_value:
                return f'Value must be less than or equal to {self._max_value}.'
        return None

    def to_dict(self) -> dict:
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'FloatArgument'}
        super_values = super(FloatArgument, self).to_dict()
        values.update(super_values)
        self._add_key_value('min_value', self._min_value, values)
        self._add_key_value('max_value', self._max_value, values)
        return values
